/* This is y4vm.c
   A part of the Y4vm library
   Copyright (C) 1998 Daniel Spangberg
   */

/*
  $Log$
  Revision 1.2  2004/05/12 15:01:40  daniels
  Updates for x86_64.

  Revision 1.1.1.1  2001/10/30 13:52:34  daniels
  Initial checkin

  Revision 1.36  1998/11/11 20:54:26  daniels
  Added new external interface.

  Revision 1.35  1998/10/30 00:38:39  daniels
  Fixed bugs in local variables. Added __call to function.

  Revision 1.34  1998/10/29 01:42:46  daniels
  Modified some bugs. More remains.

  Revision 1.33  1998/10/29 01:31:18  daniels
  Added more flexible function management.

  Revision 1.32  1998/08/10 20:32:35  daniels
  Corrected bugs in vm_free. Added statistics about dynamic memory usage.

  Revision 1.31  1998/08/10 20:02:57  daniels
  Added dynamic memory handling.

  Revision 1.30  1998/06/26 11:54:34  daniels
  Allows any data to be stored in variable.

  Revision 1.29  1998/06/23 10:49:09  daniels
  *** empty log message ***

  Revision 1.28  1998/06/23 10:48:34  daniels
  *** empty log message ***

  Revision 1.27  1998/06/22 15:41:50  daniels
  *** empty log message ***

  Revision 1.26  1998/06/22 15:39:46  daniels
  *** empty log message ***

  Revision 1.25  1998/05/24 14:24:46  daniels
  Added possibility to store pointers to variables in variables.

  Revision 1.24  1998/05/24 12:40:19  daniels
  *** empty log message ***

  Revision 1.23  1998/04/20 23:59:11  daniels
  Added call

  Revision 1.22  1998/04/14 18:37:42  daniels
  Forgot vm_relative_address... =:-|

  Revision 1.21  1998/04/13 21:18:02  daniels
  Added itos and strcat

  Revision 1.20  1998/04/13 20:26:06  daniels
  Fixed hw interrupts

  Revision 1.19  1998/04/06 23:33:28  daniels
  Added interrupts

  Revision 1.18  1998/03/22 18:39:44  daniels
  Added vm_get_variable_value

  Revision 1.17  1998/03/03 18:05:02  daniels
  Added working strcpy. Better string management.

  Revision 1.16  1998/02/08 13:45:10  daniel
  *** empty log message ***

  Revision 1.15  1998/02/06 20:01:02  daniel
  Added function vm_invalidate

  Revision 1.14  1998/02/05 17:50:28  daniel
  Changed all calls to index to strchr. index is BSD call,
  strchr conforms to POSIX

  Revision 1.13  1998/02/03 14:39:22  daniel
  Added some debugging functions, and modified how string works

  Revision 1.12  1998/02/02 20:35:28  daniel
  Made vm safer. Added $ function

  Revision 1.11  1998/02/01 19:39:11  daniel
  Added arrays

  Revision 1.10  1998/02/01 18:51:47  daniel
  Fixed local variable bug

  Revision 1.9  1998/02/01 18:39:29  daniel
  *** empty log message ***

  Revision 1.8  1998/01/31 22:49:22  daniel
  Added variables

  Revision 1.7  1998/01/31 21:27:26  daniel
  Added comments

  Revision 1.6  1998/01/31 20:05:17  daniel
  vm_get_string_from_code is global

  Revision 1.5  1998/01/31 18:02:18  daniel
  Fixed bugs in parser. Made machine even more extensible by allowing
  multiple external handlers

  Revision 1.4  1998/01/31 13:59:13  daniel
  Fixed bug in parser

  Revision 1.3  1998/01/30 22:43:34  daniel
  Added lotsa stuff

  Revision 1.2  1998/01/30 18:36:39  daniel
  Created the virtual machine as an extensible virtual machine

  Revision 1.1  1998/01/30 09:42:24  daniel
  Initial revision

  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "y4vm.h"

typedef struct
{
  char *name;
  unsigned int instr;
} i_table;

static i_table my_table[]=
{
  {"dup",I_DUP},
  {"drop",I_DROP},
  {"+",I_PLUS},
  {"-",I_MINUS},
  {"emit",I_EMIT},
  {"stop",I_STOP},
  {"nop",I_NOP},
  {"stat",I_STAT},
  {"list",I_LIST},
  {"__ret",I_RET},
  {"*",I_MUL},
  {"/",I_DIV},
  {".",I_POINT},
  {"do",I_DO},
  {"loop",I_LOOP},
  {"r",I_R},
  {">r",I_G_R},
  {"r>",I_R_G},
  {">",I_GT},
  {"<",I_LT},
  {"=",I_EQ},
  {">=",I_GTE1},
  {"=>",I_GTE2},
  {"<=",I_LTE1},
  {"=<",I_LTE2},
  {"!=",I_NE},
  {"printf",I_PRINTF},
  {"begin",I_BEGIN},
  {"until",I_UNTIL},
  {"swap",I_SWAP},
  {"@",I_LOAD},
  {"!",I_STORE},
  {"__leave",I_LEAVE},
  {"mstat",I_MSTAT},
  {"cdump",I_CODEDUMP},
  {"cast",I_CAST},
  {"string",I_STRING},
  {"strcpy",I_STRCPY},
  {"__int",I_INT},
  {"__iret",I_IRET},
  {"install",I_INSTALL},
  {"uninstall",I_DEINSTALL},
  {"__stop",I__STOP},
  {"itos",I_ITOS},
  {"strcat",I_STRCAT},
  {"__call",I_CALL},
  {"malloc",I_MALLOC},
  {"free",I_FREE},
  {"NULL",I_NULL},
  {NULL,0}
};

/* Hooks for the virtual machine */
static vm_instr *vm_code_segment=NULL;
static vm_data *vm_stack_segment=NULL;
static vm_data *vm_rstack_segment=NULL;
static vm_data *vm_data_segment=NULL;

static unsigned int vm_code_size;
static unsigned int vm_stack_size;
static unsigned int vm_rstack_size;
static unsigned int vm_data_size;

/* Program counter, stack pointer, return stack pointer */
static unsigned int vm_pc;
static unsigned int vm_sp;
static unsigned int vm_rsp;

/* Current data length */
static unsigned int vm_data_len;

/* Current code length */
static unsigned int vm_code_len;

/* Local storage */
static unsigned int vm_local_storage[MAX_NESTED_STATEMENTS];

/* Relative addressing */
static unsigned int vm_relative_address;

static int in_function_declaration=0;
static int n_vars_in_function;

static unsigned int nticks;

static int vm_running=0;
static int valid_machine=0;

static unsigned int nested[MAX_NESTED_STATEMENTS];
static unsigned int nested_var[MAX_NESTED_STATEMENTS];
static unsigned int nested_rel[MAX_NESTED_STATEMENTS];
static unsigned char nested_type[MAX_NESTED_STATEMENTS];
static int current_nest;
static char comstring[4096]; /* Should be more than enough */

typedef struct derived_type
{
  void (*destructor)(void *this);
  int this_type_id;
  int total_size;
  int elements;
  char **names;
  int *type_id; /* Type id = 0 is ordinary variable */
  int *type_length;
  struct derived_type **type_ptr;
} derived_type;

typedef struct dtype_link_t
{
  char *name;
  int type_id;
  int type_length;
  derived_type *type_ptr;
  struct dtype_link_t *next;
} dtype_link_t;

static int in_der_type_dec=0;
static derived_type *current_derived_type;
static dtype_link_t *dtype_link;
static int number_of_derived_types=0;

static void derived_type_destructor(void *thisvoid)
{
    derived_type *this=thisvoid;
    int i;
    for (i=0; i<this->elements; i++)
    {
	free(this->names[i]);
    }
    free(this->names);
    free(this->type_id);
    free(this->type_ptr);
    free(this);
    number_of_derived_types--;
}

typedef union
{
  void (*destructor)(void *this);
  derived_type data;
} general_t;

typedef struct mywords_t
{
  char *name;
  int type;
  general_t *gen;
  struct mywords_t *next;
} mywords_t;

static mywords_t *mywords=NULL;

static void register_word(char *mystr,int type,general_t *ptr)
{
  mywords_t *new=malloc(sizeof(mywords_t));
  new->next=mywords;
  mywords=new;
  new->type=type;
  new->name=malloc(strlen(mystr)+1);
  new->gen=ptr;
  strcpy(new->name,mystr);
#if 0
  printf("Registered word '%s' as %d.\n",mystr,type);
#endif
}

static void unregister_word(char *mystr,int type)
{
  mywords_t *ptr=mywords,*tmp=NULL;
  while ((ptr!=NULL) && ((strcmp(ptr->name,mystr)!=0) || (ptr->type!=type)))
    {
      tmp=ptr;
      ptr=ptr->next;
    }
  if (ptr==NULL)
    {
      char t[200];
      sprintf(t,"Fatal: Cannot remove a word: '%s' of type %d.\n",mystr,type);
      vmputs(t);
      exit(1);
    }
  if (ptr->gen!=NULL)
    ptr->gen->destructor(ptr->gen);

  free(ptr->name);
  if (tmp==NULL)
    mywords=ptr->next;
  else
    tmp->next=ptr->next;
#if 0
  printf("Unregistered word '%s' as %d.\n",mystr,type);
#endif
}

static void unregister_all_words()
{
  mywords_t *ptr=mywords,*tmp=NULL;
  while (ptr!=NULL)
    {
      tmp=ptr;
      free(ptr->name);
      ptr=ptr->next;
      free(tmp);
    }
  mywords=NULL;
}

static int determine_word_type(char *mystr,int len,void **gen)
{
  mywords_t *ptr=mywords;
  if (len==0)
    return 0; /* Unknown word */
  while ((ptr!=NULL) && ((strlen(ptr->name)!=len) || (strncmp(mystr,ptr->name,len)!=0)))
    ptr=ptr->next;
  if (ptr==NULL)
    return(0); /* Unknown word */
  *gen=ptr->gen;
  return(ptr->type);
}

#define MAX_INTERRUPTS 16

typedef struct vm_interrupt
{
  unsigned int addr;
  struct vm_interrupt *next;
} vm_interrupt;

static vm_interrupt *int_vector[MAX_INTERRUPTS];
static int interrupt_flag=1;

typedef struct memory_list_t
{
  unsigned int start;
  unsigned int size;
  vm_data *address;
  struct memory_list_t *next;
} memory_list_t;

static memory_list_t *memory_list=NULL;
static unsigned int vm_dynamic_memory_amount=0;
static unsigned int vm_dynamic_memory_blocks=0;

static unsigned int vm_malloc(unsigned int size)
{
  /* Find first unallocated memory address that can fit a block of size size */
  memory_list_t *ptr=memory_list, *new;
  int found=0;
  while (!found)
    {
      if (ptr->next==NULL)
	found=1;
      else
	{
	  unsigned int this_start=ptr->start+ptr->size;
	  unsigned int this_end=ptr->next->start;
	  if (size<=(this_end-this_start))
	    found=1;
	  else
	    ptr=ptr->next;
	}
    }
  new=malloc(sizeof(memory_list_t));
  if (new==NULL)
    return VM_NULL_PTR;
  new->start=ptr->start+ptr->size;
  new->size=size;
  new->address=malloc(sizeof(vm_data)*size);
  if (new->address==NULL)
    return VM_NULL_PTR;
  new->next=ptr->next;
  ptr->next=new;
  vm_dynamic_memory_amount+=size;
  vm_dynamic_memory_blocks++;
  return new->start;
}

static void vm_free(unsigned int free_ptr)
{
  if (free_ptr!=VM_NULL_PTR)
    {
      memory_list_t *ptr=memory_list, *tmp=NULL;
      while ((ptr!=NULL) && (ptr->start!=free_ptr))
	{
	  tmp=ptr;
	  ptr=ptr->next;
	}
      if (ptr==NULL)
	{
	  vmputs("Cannot free unallocated memory.\n");
	  vm_stop();
	}
      else
	{
	  vm_dynamic_memory_amount-=ptr->size;
	  vm_dynamic_memory_blocks--;
	  tmp->next=ptr->next;
	  free(ptr->address);
	  free(ptr);
	}
    }
}

static void vm_i_malloc()
{
  if (test_pop_stack("malloc"))
    {
      vm_data data=pop_stack();
      int size=(unsigned int)(data.data.mydouble+0.5);
      data.type=DATA_POINTER_TO_DATA;
      data.data.myvarp.address=vm_malloc(size);
      data.data.myvarp.size=size;
      data.data.myvarp.offset=0;
      data.data.myvarp.local=0;
      push_stack(data);
    }
}

static void vm_i_free()
{
  if (test_pop_stack("free"))
    {
      vm_data data=pop_stack();
      if (data.type!=DATA_POINTER_TO_DATA)
	{
	  vmputs("free expected pointer on stack.\n");
	  vm_stop();
	}
      else
	vm_free(data.data.myvarp.address);
    }
}

static vm_data *vm_address(unsigned int address)
{
  memory_list_t *ptr=memory_list;
  while ((ptr!=NULL) && (address>=(ptr->start+ptr->size)))
    ptr=ptr->next;
  if (ptr==NULL)
    {
      vmputs("Segmentation violation in vm.\n");
      vm_stop();
      return (NULL);
    }
  return(ptr->address+address-ptr->start);
}

static void vm_free_dynamic_memory()
{
  memory_list_t *ptr=memory_list->next,*tmp;
  while (ptr!=NULL)
    {
      free(ptr->address);
      tmp=ptr;
      ptr=ptr->next;
      free(tmp);
    }
  memory_list->next=NULL;
}

static void register_default_words()
{
  i_table *i_p=my_table;
  /* Register all the word types ( more to come when program is running ) */
  while (i_p->name!=NULL)
    {
      register_word(i_p->name,WORD_INTERNAL,NULL);
      i_p++;
    }
  register_word("$",WORD_DOLLAR,NULL);
  register_word("var",WORD_VARDEC,NULL);
  register_word(":",WORD_COLON,NULL);
  register_word(";",WORD_SEMICOLON,NULL);
  register_word("if",WORD_IF,NULL);
  register_word("else",WORD_ELSE,NULL);
  register_word("then",WORD_THEN,NULL);
  register_word("type",WORD_TYPE,NULL);

  /* Other words known but handled as special cases:
     ( (comment)
     ' (string)
     floats
   */
}

void vm_init()
{
  memory_list_t *mem_new;
  int i;
  vm_code_size=DEFAULT_INSTR_SIZE;
  vm_stack_size=DEFAULT_STACK_SIZE;
  vm_rstack_size=DEFAULT_RSTACK_SIZE;
  vm_data_size=DEFAULT_DATA_SIZE;
  vm_code_segment=malloc(sizeof(vm_instr)*vm_code_size);
  vm_stack_segment=malloc(sizeof(vm_data)*vm_stack_size);
  vm_rstack_segment=malloc(sizeof(vm_data)*vm_rstack_size);
  vm_data_segment=malloc(sizeof(vm_data)*vm_data_size);
  mem_new=malloc(sizeof(memory_list_t));
  mem_new->next=memory_list;
  mem_new->start=0;
  mem_new->size=vm_data_size;
  mem_new->address=vm_data_segment;
  memory_list=mem_new;
  for (i=0; i<MAX_INTERRUPTS; i++)
    int_vector[i]=NULL;
  interrupt_flag=1;
  register_default_words();
}

static void add_interrupt(int nr,unsigned int addr)
{
  if ((nr>=0) && (nr<MAX_INTERRUPTS))
    {
      vm_interrupt *new=malloc(sizeof(vm_interrupt));
      new->next=int_vector[nr];
      new->addr=addr;
      int_vector[nr]=new;
    }
}

static void remove_interrupt(int nr,unsigned int addr)
{
  if ((nr>=0) && (nr<MAX_INTERRUPTS))
    {
      vm_interrupt *ptr=int_vector[nr],*tmp=NULL;
      while ((ptr!=NULL) && (ptr->addr!=addr))
	{
	  tmp=ptr;
	  ptr=ptr->next;
	}
      if ((ptr!=NULL) && (ptr->addr==addr))
	{
	  if (tmp==NULL)
	    {
	      int_vector[nr]=ptr->next;
	    }
	  else
	    {
	      tmp->next=ptr->next;
	    }
	  free(ptr);
	}
    }
}

static void remove_all_interrupts()
{
  int i;
  for (i=0; i<MAX_INTERRUPTS; i++)
    {
      vm_interrupt *ptr=int_vector[i],*tmp;
      while(ptr!=NULL)
	{
	  tmp=ptr;
	  ptr=ptr->next;
	  free(tmp);
	}
      int_vector[i]=NULL;
    }  
}

typedef struct f_table
{
  char *name;
  unsigned int f_pc;
  struct f_table *next;
} f_table;

typedef struct e_table
{
  char *name;
  unsigned int handle;
  void (*external_word)(void);
  struct e_table *next;
} e_table;

typedef struct var_table
{
  char *name;
  unsigned int address;
  unsigned int size;
  int type;
  derived_type *ptr;
  char local;
  struct var_table *next;
} var_table;

typedef struct e_handler
{
  void (*external_interface)(unsigned int);  
  struct e_handler *next;
} e_handler;

static e_handler *my_handler_hook=NULL;

static void clear_interface_table()
{
  e_handler *ptr=my_handler_hook,*tmp;
  while(ptr!=NULL)
    {
      tmp=ptr;
      ptr=ptr->next;
      free(tmp);
    }
  my_handler_hook=NULL;
}

static f_table *my_f_table=NULL;

static void clear_f_table()
{
  f_table *ptr=my_f_table,*tmp;
  while(ptr!=NULL)
    {
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_FUNCTION);
      free(tmp->name);
      free(tmp);
    }
  my_f_table=NULL;
}

static void forget_function(unsigned int address)
{
  f_table *ptr=my_f_table,*tmp;
  unsigned int last_address;
  while ((ptr!=NULL) && (ptr->f_pc>=address))
    {
      last_address=ptr->f_pc;
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_FUNCTION);
      free(tmp->name);
      free(tmp);
    }
  my_f_table=ptr;
  if (last_address!=address)
    {
      vmputs("Horribly corrupted code segment (please reset)\n");
      vm_stop();
    }
  vm_code_len=address-1;
  vm_pc=vm_code_len;
  save_vm_state();
  vm_stop();
}

static void forget_function_names_after(unsigned int address)
{
  f_table *ptr=my_f_table,*tmp;
  unsigned int last_address;
  while ((ptr!=NULL) && (ptr->f_pc>=address))
    {
      last_address=ptr->f_pc;
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_FUNCTION);
      free(tmp->name);
      free(tmp);
    }
  my_f_table=ptr;
}

static void forget_last_function()
{
  unsigned int a;
  f_table *ptr=my_f_table,*tmp;
  if (ptr!=NULL)
    {
      a=ptr->f_pc;
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_FUNCTION);
      free(tmp->name);
      free(tmp);
      my_f_table=ptr;
    }
  else
    {
      vmputs("Horribly corrupted code segment (please reset)\n");
      vm_stop();
    }
  vm_code_len=a-1;
  vm_pc=vm_code_len;
  save_vm_state();
  vm_stop();
}

static e_table *my_e_table=NULL;
static unsigned int e_handle=0;
static void (*vm_send_handler)(char *);

static void clear_e_table()
{
  e_table *ptr=my_e_table,*tmp;
  while(ptr!=NULL)
    {
      tmp=ptr;
      ptr=ptr->next;
      free(tmp->name);
      free(tmp);
    }
  my_e_table=NULL;
  e_handle=0;
  clear_interface_table();
  vm_send_handler=NULL;
}

static var_table *my_var_table=NULL;

static void clear_var_table()
{
  var_table *ptr=my_var_table,*tmp;
  while(ptr!=NULL)
    {
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_VARIABLE);
      free(tmp->name);
      free(tmp);
    }
  my_var_table=NULL;
}

static void remove_n_variables(int nvars)
{
  var_table *ptr=my_var_table,*tmp;
  while((ptr!=NULL) && (nvars>0))
    {
      tmp=ptr;
      ptr=ptr->next;
      unregister_word(tmp->name,WORD_VARIABLE);
      free(tmp->name);
      free(tmp);
      nvars--;
    }
  my_var_table=ptr;
  if (nvars>0)
    {
      vmputs("Horribly corrupted variable list (please reset)\n");
      vm_stop();
    }
}

static void add_variable_to_table(char *name,int type,derived_type *ptr,unsigned int address,unsigned int size,int local_var)
{
  var_table *new=malloc(sizeof(var_table));
  new->name=malloc(strlen(name)+1);
  strcpy(new->name,name);
  new->address=address;
  new->size=size;
  new->local=local_var;
  new->type=type;
  new->ptr=ptr;
  new->next=my_var_table;
  my_var_table=new;
  register_word(name,WORD_VARIABLE,NULL);
}

double vm_get_variable_value(char *name,int offset)
{
  /* Search for variable */
  var_table *ptr;
  int mylen=strlen(name);
  ptr=my_var_table;
  while((ptr!=NULL) &&
	((strlen(ptr->name)!=mylen) ||
	 (strncmp(ptr->name,name,(size_t)mylen)!=0)))
    ptr=ptr->next;
  if ((ptr!=NULL) && (offset<ptr->size) &&(offset>=0))
    return(vm_address(ptr->address+offset)->data.mydouble);
  else
    return 0;
}

static void add_function(char *name,unsigned int f_pc)
{
  f_table *new=malloc(sizeof(f_table));
  new->name=malloc(strlen(name)+1);
  strcpy(new->name,name);
  new->f_pc=f_pc;
  new->next=my_f_table;
  my_f_table=new;
  register_word(name,WORD_FUNCTION,NULL);
}

unsigned int add_external_word(char *name)
{
  e_table *new=malloc(sizeof(e_table));
  new->name=malloc(strlen(name)+1);
  strcpy(new->name,name);
  e_handle++;
  new->handle=e_handle;
  new->next=my_e_table;
  my_e_table=new;
  register_word(name,WORD_EXTERNAL,NULL);
  return(e_handle);
}

void add_y4_word(char *name, void (*function)(void))
{
  e_table *new=malloc(sizeof(e_table));
  new->name=malloc(strlen(name)+1);
  strcpy(new->name,name);
  new->external_word=function;
  new->next=my_e_table;
  my_e_table=new;
  register_word(name,WORD_NEW_EXTERNAL,NULL);
}

void add_external_interface(void (*ehandler)(unsigned int handle))
{
  e_handler *new=malloc(sizeof(e_handler));
  new->external_interface=ehandler;
  new->next=my_handler_hook;
  my_handler_hook=new;
}

static void call_external_function(unsigned int handle)
{
  e_handler *ptr=my_handler_hook;
  while ((valid_machine) && (ptr!=NULL))
    {
      ptr->external_interface(handle);
      ptr=ptr->next;
    }
}

void add_vm_send_data_handler(void (*esend)(char *str))
{
  vm_send_handler=esend;
}

void vmputs(char *str)
{
  if (vm_send_handler!=NULL)
    vm_send_handler(str);
}

void vm_clear()
{
  int i;
  vm_instr my_instr;
  vm_pc=0;
  vm_sp=0;
  vm_rsp=0;
  vm_code_len=0;
  vm_data_len=VM_BASE_OF_MEMORY;
  nticks=0;
  current_nest=0;
  valid_machine=0;
  clear_e_table();
  clear_f_table();
  clear_var_table();
  unregister_all_words();
  register_default_words();
  vm_free_dynamic_memory();
  /* Fill code segment with STOP instructions */
  my_instr.type=INSTR_INTERNAL;
  my_instr.data.myint=I_STOP;
  for (i=0; i<vm_code_size; i++)
    vm_code_segment[i]=my_instr;
  remove_all_interrupts();
  interrupt_flag=1;
}

void vm_deinit()
{
  vm_clear();
  valid_machine=0;
  free(vm_code_segment);
  free(vm_stack_segment);
  free(vm_rstack_segment);
  free(vm_data_segment);
  unregister_all_words();
}

static char *oknumberchar="0123456789";

static int is_a_number(char n)
{
  if (strchr(oknumberchar,n)!=NULL)
    return 1;
  else
    return 0;
}

static char *okfloatchar="0123456789.eE+-";

static int instr_is_float(char *begin,char *end)
{
  int okfloat=1;
  int hasanumber=0;
  while((begin<=end) && (strchr(okfloatchar,*begin)!=NULL))
    {
      if (is_a_number(*begin))
	hasanumber=1;
      begin++;
    }
  if (begin<=end)
    okfloat=0;
  if (!hasanumber)
    okfloat=0;
  return okfloat;
}

static char *okintchar="0123456789+-";

static int instr_is_int(char *begin,char *end)
{
  int okint=1;
  int hasanumber=0;
  while((begin<=end) && (strchr(okintchar,*begin)!=NULL))
    {
      if (is_a_number(*begin))
	hasanumber=1;
      begin++;
    }
  if (begin<=end)
    okint=0;
  if (!hasanumber)
    okint=0;
  return okint;
}

static char *whitespace=" \t\n";

void vm_parse_and_compile(char *str)
{
  char *begin=str;
  char *end;
  int pstrlen=strlen(str);

  while((vm_running) && (*begin!='\0'))
    {
      /* Search for non whitespace */
      while((*begin!='\0') && (strchr(whitespace,*begin)!=NULL))
	begin++;
      if (*begin!='\0')
	{
	  i_table *my_db;
	  f_table *my_f_db;
	  e_table *my_e_db;
	  var_table *my_var_db;
	  if (vm_code_len>vm_code_size)
	    {
	      sprintf(comstring,"Code size exceeded\n");
	      vmputs(comstring);
	      return;
	    }
	  end=begin;
	  /* Search for whitespace */
	  while(strchr(whitespace,*end)==NULL)
	    end++;
	  /* Specially handled instructions */
	  /* Check if float */
	  if ((instr_is_float(begin,end-1)) && !in_der_type_dec)
	    {
	      /* Float */
	      char myf[64];
	      int len=end-begin;
	      if (len>63) 
		{
		  sprintf(comstring,"Warning: float truncated\n");
		  vmputs(comstring);
		  len=63;
		}
	      memcpy(myf,begin,len);
	      myf[len]='\0';
	      vm_code_segment[vm_code_len].type=INSTR_FLOAT;
	      vm_code_segment[vm_code_len].data.mydouble=atof(myf);
	      vm_code_len++;
	    }
	  else
	    {
	      if ((strncmp("'",begin,1)==0) && !in_der_type_dec)
		{
		  /* String */
		  /* Strings are nasty bastards in this vm.
		     They are stored one byte in each machine instruction. So we are wasting
		     lots and lots of space. But I don't expect that many strings in 
		     this machine anyway. */
		  int vm_jinstr=vm_code_len;
		  vm_code_segment[vm_code_len++].type=INSTR_JUMP;
		  begin++;
		  while((begin<=(str+pstrlen)) && (*begin!='\''))
		    {
		      unsigned char d=*begin++;
		      vm_code_segment[vm_code_len].type=INSTR_DATA;
		      if (d=='\\')
			{
			  if (begin<(str+pstrlen))
			    {
			      unsigned char d2=*begin++;
			      switch((int)d2)
				{
				case 'n':
				  d='\n';
				  break;
				case 't':
				  d='\t';
				  break;
				}
			    }
			}
		      vm_code_segment[vm_code_len++].data.myint=d;
		    }
		  if (*begin=='\'')
		    {
		      /* EOS */
		      vm_code_segment[vm_code_len].type=INSTR_DATA;
		      vm_code_segment[vm_code_len++].data.myint=0;
		      /* Address to instr after string */
		      vm_code_segment[vm_jinstr].data.myint=vm_code_len;
		      /* Instruction that pushes pointer to string on stack */ 
		      vm_code_segment[vm_code_len].type=INSTR_POINTER_TO_STRING;
		      vm_code_segment[vm_code_len++].data.myint=vm_jinstr+1;
		    }
		  else
		    {
		      sprintf(comstring,"Missing end of string\n");
		      vmputs(comstring);
		      vm_stop();
		    }
		  end=begin+1;
		}
	      else
		{
		  if (((int)(end-begin)==1) && (strncmp("(",begin,(size_t)(end-begin))==0))
		    {
		      /* Comment */
		      /* Search for ) but count any extra ( */
		      /* Nested comments are allowed */
		      int np=1;
		      begin++;
		      while((begin<=(str+pstrlen)) && (np!=0))
			{
			  if (*begin=='(')
			    np++;
			  if (*begin==')')
			    np--;
			  begin++;
			}
		      if (np!=0)
			{
			  vmputs("Missing end of comment )\n");
			  vm_stop();
			}
		      end=begin+1;
		    }
		  else
		    {
		      int this_type;
		      void *ptr_to_data;
		      switch(this_type=determine_word_type(begin,(int)(end-begin),&ptr_to_data))
			{
			case 0:
			  {
			    char *myf=malloc(((int)(end-begin+1)));
			    memcpy(myf,begin,(int)(end-begin));
			    myf[(int)(end-begin)]='\0';
			    sprintf(comstring,"I don't know what %s means\n",myf);
			    vmputs(comstring);
			    vm_stop();
			  }
			  break;
			case WORD_DOLLAR:
			  {
			    /* Search for function name */
			    begin=end;
			    /* Search for non whitespace */
			    while(strchr(whitespace,*begin)!=NULL)
			      begin++;
			    end=begin;
			    /* Search for whitespace */
			    while(strchr(whitespace,*end)==NULL)
			      end++;
			    if ((end==begin) || in_der_type_dec)
			      {
				sprintf(comstring,"Parse error in address of function\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				char *pos;
				unsigned int address,size;
				char *fname=malloc(((int)(end-begin)+1));
				strncpy(fname,begin,(size_t)(end-begin));
				fname[(int)(end-begin)]='\0';
				/* Search for function */
				my_f_db=my_f_table;
				while((my_f_db!=NULL) &&
				      ((strlen(my_f_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_f_db->name,begin,(size_t)(end-begin))!=0)))
				  my_f_db=my_f_db->next;
				if (my_f_db!=NULL)
				  {
				    /* Function address */
				    vm_code_segment[vm_code_len].type=INSTR_POINTER_TO_FUNCTION;
				    vm_code_segment[vm_code_len].data.myint=my_f_db->f_pc;
				    vm_code_len++;
				  }
				else
				  {
				    vmputs("Invalid function name in address of function\n");
				    vm_stop();
				  }
			      }
			  }
			  break;
			case WORD_TYPE:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Nested type declaration is not allowed\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				/* Variable declaration */
				begin=end;
				/* Search for non whitespace */
				while(strchr(whitespace,*begin)!=NULL)
				  begin++;
				end=begin;
				/* Search for whitespace */
				while(strchr(whitespace,*end)==NULL)
				  end++;
				if (end==begin)
				  {
				    sprintf(comstring,"Parse error in type definition\n");
				    vmputs(comstring);
				    vm_stop();
				  }
				else
				  {
				    unsigned int address,size;
				    char *fname=malloc(((int)(end-begin)+1));
				    derived_type *new=malloc(sizeof(derived_type));
				    new->destructor=derived_type_destructor;
				    strncpy(fname,begin,(size_t)(end-begin));
				    fname[(int)(end-begin)]='\0';
				    nested_type[current_nest]=N_TYPE;
				    current_nest++;
				    number_of_derived_types++;
				    new->this_type_id=number_of_derived_types;
				    new->elements=0;
				    new->total_size=0;
				    current_derived_type=new;

				    register_word(fname,WORD_DERIVED_TYPE,(general_t*)new);

				    dtype_link=NULL;
				    in_der_type_dec=1;
				  }
			      }
			  }
			  break;
			case WORD_VARDEC:
			case WORD_DERIVED_TYPE:
			  {
			    /* Variable declaration */
			    begin=end;
			    /* Search for non whitespace */
			    while(strchr(whitespace,*begin)!=NULL)
			      begin++;
			    end=begin;
			    /* Search for whitespace */
			    while(strchr(whitespace,*end)==NULL)
			      end++;
			    if (end==begin)
			      {
				sprintf(comstring,"Parse error in variable declaration\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				char *pos;
				unsigned int address,size;
				char *fname=malloc(((int)(end-begin)+1));
				strncpy(fname,begin,(size_t)(end-begin));
				fname[(int)(end-begin)]='\0';
				/* Is this an array?? */
				pos=strchr(fname,'[');
				if (pos!=NULL)
				  {
				    /* Yes an array */
				    char *pos2=strchr(pos,']');
				    if (pos2==NULL)
				      {
					vmputs("Malformed array declaration\n");
					vm_stop();
				      }
				    else
				      {
					/* Check if integer */
					if (instr_is_int(pos+1,pos2-1))
					  {
					    /* Int */
					    char myf[64];
					    int len=pos2-pos-1;
					    if (len>63)
					      {
						sprintf(comstring,"Warning: Array size truncated\n");
						vmputs(comstring);
						len=63;
					      }
					    memcpy(myf,pos+1,len);
					    myf[len]='\0';
					    size=atoi(myf);
					  }
					else
					  {
					    vmputs("Invalid array size\n");
					    size=0;
					    vm_stop();
					  }
					*pos='\0';
				      }
				  }
				else
				  {
				    /* No not an array */
				    size=1;
				  }
				if (size<1)
				  {
				    vmputs("Array size must be positive\n");
				    vm_stop();
				  }
				else
				  {
				    if (this_type==WORD_DERIVED_TYPE)
				      {
					size*=((derived_type *)ptr_to_data)->total_size;
				      }
				    if (in_der_type_dec)
				      {
					dtype_link_t *new=malloc(sizeof(dtype_link_t));
					new->next=dtype_link;
					dtype_link=new;
					new->name=malloc(strlen(fname)+1);
					strcpy(new->name,fname);
					if (this_type==WORD_DERIVED_TYPE)
					  {
					    new->type_id=((derived_type *)ptr_to_data)->this_type_id;
					    new->type_ptr=(derived_type *)ptr_to_data;
					  }
					else
					  new->type_id=0;
					new->type_length=size;
					current_derived_type->total_size+=size;
					current_derived_type->elements++;
				      }
				    else
				      {
					int var_type=0;
					derived_type *var_ptr=NULL;
					if (this_type==WORD_DERIVED_TYPE)
					  {
					    var_type=((derived_type *)ptr_to_data)->this_type_id;
					    var_ptr=(derived_type *)ptr_to_data;
					  }
					if (in_function_declaration)
					  {
					    /* Local variable */
					    address=vm_relative_address;
					    add_variable_to_table(fname,var_type,var_ptr,address,size,in_function_declaration);
					    vm_local_storage[in_function_declaration-1]+=size;
					    vm_relative_address+=size;
					    n_vars_in_function++;
					    /* Also emit instructions to allocate the storage for this
					       variable */
					    vm_code_segment[vm_code_len].type=INSTR_VARALLOT;
					    vm_code_segment[vm_code_len++].data.myint=size;
					  }
					else
					  {
					    /* Global variable */
					    add_variable_to_table(fname,var_type,var_ptr,vm_data_len,size,0);
					    if ((vm_data_len+size)>vm_data_size)
					      {
						vmputs("Out of data memory on variable allocation\n");
						vm_stop();
					      }
					    else
					      vm_data_len+=size;
					  }
				      }
				  }
				free(fname);
			      }
			  }
			  break;
			case WORD_COLON:
			  {
			    /* Function declaration */
			    begin=end;
			    /* Search for non whitespace */
			    while(strchr(whitespace,*begin)!=NULL)
			      begin++;
			    end=begin;
			    /* Search for whitespace */
			    while(strchr(whitespace,*end)==NULL)
			      end++;
			    if ((end==begin) || in_der_type_dec)
			      {
				sprintf(comstring,"Parse error in function declaration\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				char *fname=malloc(((int)(end-begin)+1));
				strncpy(fname,begin,(size_t)(end-begin));
				fname[(int)(end-begin)]='\0';
				/* printf("Adding function %s at %d\n",fname,vm_code_len); */
				add_function(fname,vm_code_len+1);
				free(fname);
				in_function_declaration++;
				  
				/* Nested instruction */
				vm_code_segment[vm_code_len].type=INSTR_JUMP;
				/* Put in database */
				nested[current_nest]=vm_code_len;
				nested_var[current_nest]=n_vars_in_function;
				nested_rel[current_nest]=vm_relative_address;
				nested_type[current_nest]=N_FN;
				current_nest++;
				vm_code_len++;
				  
				vm_relative_address=0;
				n_vars_in_function=0;

				vm_code_segment[vm_code_len].type=INSTR_ENTER;
				vm_code_segment[vm_code_len++].data.myint=in_function_declaration;
			      }
			  }
			  break;
			case WORD_SEMICOLON:
			  {
			    if (in_function_declaration==0)
			      {
				if (in_der_type_dec)
				  {
				    if (nested_type[current_nest-1]!=N_TYPE)
				      {
					vmputs("Invalid nesting: ; doesn't match type\n");
					vm_stop();
				      }
				    else
				      {
					int i;
					dtype_link_t *ptr=dtype_link,*tmp;
					/* Move over temporary information in linked lists */
					current_derived_type->names=
					  malloc(sizeof(char*)*current_derived_type->elements);
					current_derived_type->type_id=
					  malloc(sizeof(int)*current_derived_type->elements);
					current_derived_type->type_length=
					  malloc(sizeof(int)*current_derived_type->elements);
					current_derived_type->type_ptr=
					  malloc(sizeof(derived_type*)*current_derived_type->elements);
					for (i=0; i<current_derived_type->elements; i++)
					  {
					    current_derived_type->names[i]=malloc(strlen(ptr->name)+1);
					    strcpy(current_derived_type->names[i],ptr->name);
					    free(ptr->name);
					    current_derived_type->type_id[i]=ptr->type_id;
					    current_derived_type->type_length[i]=ptr->type_length;
					    current_derived_type->type_ptr[i]=ptr->type_ptr;
					    tmp=ptr;
					    ptr=ptr->next;
					    free(tmp);
					  }
					in_der_type_dec=0;
					current_nest--;
				      }
				  }
				else
				  {
				    vmputs("Warning: Use of ; outside function declaration\n");
				  }
			      }
			    else
			      {
				/* printf("Ending function declaration at %d\n",vm_code_len); */
				/* Internal instruction */
				vm_code_segment[vm_code_len].type=INSTR_INTERNAL;
				vm_code_segment[vm_code_len++].data.myint=I_LEAVE;
				vm_code_segment[vm_code_len].type=INSTR_INTERNAL;
				vm_code_segment[vm_code_len++].data.myint=I_RET;				      
				/* Place address to next instruction in jump before function */
				if (nested_type[current_nest-1]!=N_FN)
				  {
				    vmputs("Invalid nesting: Unmatched ;\n");
				    vm_stop();
				  }
				else
				  {
				    unsigned int oldvm_pc=vm_pc;
					  
				    /* Remove function names contained in this function.
				       The nested array contains (in this case) the address to the
				       jump instruction. The function is on the address after that one.
				       Forget everything after that function => +2 */
				    forget_function_names_after(nested[current_nest-1]+2);

				    remove_n_variables(n_vars_in_function);

				    vm_code_segment[nested[current_nest-1]].data.myint=vm_code_len;
				    /* Restore variables for this function */
				    n_vars_in_function=nested_var[current_nest-1];
				    vm_relative_address=nested_rel[current_nest-1];

				    current_nest--;
					  				      
				    /* Move program counter after function */
				    vm_pc=vm_code_len;

				    /* A one time jump to previous pc */
				    vm_code_segment[vm_code_len].type=INSTR_JUMP_ONCE;
				    vm_code_segment[vm_code_len++].data.myint=oldvm_pc;

				    in_function_declaration--;
				  }
			      }
			  }
			  break;
			case WORD_IF:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Use of IF in type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				/* Nested instruction */
				vm_code_segment[vm_code_len].type=INSTR_IF;
				/* Put in database */
				nested[current_nest]=vm_code_len;
				nested_type[current_nest]=N_IF;
				current_nest++;
				vm_code_len++;
			      }
			  }
			  break;
			case WORD_ELSE:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Use of ELSE in type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {

				/* There should be an if instruction in the list */
				if (nested_type[current_nest-1]!=N_IF)
				  {
				    sprintf(comstring,"Nesting error. else must be connected to if\n");
				    vmputs(comstring);
				    vm_stop();
				  }
				else
				  {
				/* Nested instruction */
				/* Place address to next instruction in if statement */
				    vm_code_segment[nested[current_nest-1]].data.myint=vm_code_len+1;
				    vm_code_segment[vm_code_len].type=INSTR_ELSE;
				/* Put in database */
				    nested[current_nest-1]=vm_code_len;
				    nested_type[current_nest-1]=N_ELSE;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_THEN:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Use of THEN in type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {

				/* There should be an if or an else instruction in the list */
				if ((nested_type[current_nest-1]!=N_IF) && (nested_type[current_nest-1]!=N_ELSE))
				  {
				    sprintf(comstring,"Nesting error. then must be connected to if or else\n");
				    vmputs(comstring);
				    vm_stop();
				  }
				else
				  {
				/* Nested instruction */
				/* Place address to next instruction in if statement */
				    vm_code_segment[nested[current_nest-1]].data.myint=vm_code_len+1;
				    vm_code_segment[vm_code_len].type=INSTR_THEN;
				/* Remove database */
				    current_nest--;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_INTERNAL:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Invalid use of type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {

				/* Find instruction in database */
				/* Search internal database */
				my_db=my_table;
				while((my_db->name!=NULL) &&
				      ((strlen(my_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_db->name,begin,(size_t)(end-begin))!=0)))
				  my_db++;
				if (my_db->name!=NULL)
				  {
				/* Internal instruction */
				    vm_code_segment[vm_code_len].type=INSTR_INTERNAL;
				    vm_code_segment[vm_code_len].data.myint=my_db->instr;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_FUNCTION:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Use of function in type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				/* Search for declared function */
				my_f_db=my_f_table;
				while((my_f_db!=NULL) &&
				      ((strlen(my_f_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_f_db->name,begin,(size_t)(end-begin))!=0)))
				  my_f_db=my_f_db->next;
				if (my_f_db!=NULL)
				  {
				/* Function call */
				    vm_code_segment[vm_code_len].type=INSTR_FUNCTION_CALL;
				    vm_code_segment[vm_code_len].data.myint=my_f_db->f_pc;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_VARIABLE:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Use of variable in type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {

				/* Search for variable */
				my_var_db=my_var_table;
				while((my_var_db!=NULL) &&
				      ((strlen(my_var_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_var_db->name,begin,(size_t)(end-begin))!=0)))
				  my_var_db=my_var_db->next;
				if (my_var_db!=NULL)
				  {
				/* Variable */
				    vm_code_segment[vm_code_len].type=INSTR_VARIABLE;
				    vm_code_segment[vm_code_len].data.myvarp.address=my_var_db->address;
				    vm_code_segment[vm_code_len].data.myvarp.size=my_var_db->size;
				    vm_code_segment[vm_code_len].data.myvarp.local=my_var_db->local;
				    vm_code_segment[vm_code_len].data.myvarp.offset=0;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_EXTERNAL:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Invalid use of type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {

				/* Search for external function */
				my_e_db=my_e_table;
				while((my_e_db!=NULL) &&
				      ((strlen(my_e_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_e_db->name,begin,(size_t)(end-begin))!=0)))
				  my_e_db=my_e_db->next;
				if (my_e_db!=NULL)
				  {
				/* External call */
				    vm_code_segment[vm_code_len].type=INSTR_EXTERNAL;
				    vm_code_segment[vm_code_len].data.myint=my_e_db->handle;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			case WORD_NEW_EXTERNAL:
			  {
			    if (in_der_type_dec)
			      {
				sprintf(comstring,"Invalid use of type declaration.\n");
				vmputs(comstring);
				vm_stop();
			      }
			    else
			      {
				/* Search for external function */
				my_e_db=my_e_table;
				while((my_e_db!=NULL) &&
				      ((strlen(my_e_db->name)!=(int)(end-begin)) ||
				       (strncmp(my_e_db->name,begin,(size_t)(end-begin))!=0)))
				  my_e_db=my_e_db->next;
				if (my_e_db!=NULL)
				  {
				/* External call */
				    vm_code_segment[vm_code_len].type=INSTR_NEW_EXTERNAL;
				    vm_code_segment[vm_code_len].data.myptr=my_e_db->external_word;
				    vm_code_len++;
				  }
			      }
			  }
			  break;
			}
		    }
		}
	    }
	  begin=end;
	}
    }
  /* Ok; have we forgotten an ending ; ? */
  if (in_function_declaration>0)
    {
      in_function_declaration=0;
      vmputs("Missing closing ; in function declaration\n");
      forget_last_function();
      vm_stop();
    }
  /* Missing some nested commands? */
  if (current_nest>0)
    {
      current_nest=0;
      vmputs("Missing nested statements such as else, then or ;\n");
      if (in_der_type_dec)
	{
	  in_der_type_dec=0;
	}
      vm_stop();
    }
}

static int vm_pc_saved;

void save_vm_state()
{
  vm_pc_saved=vm_pc;
}

void restore_vm_state()
{
  vm_pc=vm_pc_saved;
  vm_code_len=vm_pc;
}

void vm_run()
{
  vm_running=1;
  valid_machine=1;
}

static void vm_i__stop()
{
  vm_running=0;
}

void vm_stop()
{
  vm_i__stop();
  vmputs("ok\n");
}

void vm_invalidate()
{
  valid_machine=0;
}

int test_pop_stack(char *fn)
{
  if (vm_sp==0)
    {
      sprintf(comstring,"Error: %s expected value on stack, but stack is empty.\n",fn);
      vmputs(comstring);
      vm_stop();
    }
  return vm_running;
}

int test_push_stack(char *fn)
{
  if (vm_sp==vm_stack_size)
    {
      sprintf(comstring,"Error: Stack full. %s need space on stack.\n",fn);
      vmputs(comstring);
      vm_stop();
    }
  return vm_running;
}

static int test_pop_rstack(char *fn)
{
  if (vm_rsp==0)
    {
      sprintf(comstring,"Error: %s expected value on return stack, but return stack is empty.\n",fn);
      vmputs(comstring);
      vm_stop();
    }
  return vm_running;
}

static int test_push_rstack(char *fn)
{
  if (vm_rsp==vm_rstack_size)
    {
      sprintf(comstring,"Error: Return stack full. %s need space on return stack.\n",fn);
      vmputs(comstring);
      vm_stop();
    }
  return vm_running;
}

vm_data pop_stack()
{
  vm_sp--;
  return (vm_stack_segment[vm_sp]);
}

void push_stack(vm_data data)
{
  vm_stack_segment[vm_sp]=data;
  vm_sp++;
}

static vm_data pop_rstack()
{
  vm_rsp--;
  return (vm_rstack_segment[vm_rsp]);
}

static void push_rstack(vm_data data)
{
  vm_rstack_segment[vm_rsp]=data;
  vm_rsp++;
}


/* Internal instructions */

static void vm_i_dup()
{
  if (test_pop_stack("dup") && (test_push_stack("dup")))
    {
      vm_data data=pop_stack();
      push_stack(data);
      push_stack(data);
    }
}

static void vm_i_drop()
{
  if (test_pop_stack("drop"))
    {
      pop_stack();
    }
}

static void vm_i_swap()
{
  if (test_pop_stack("swap"))
    {
      vm_data data1=pop_stack();
      if (test_pop_stack("swap"))
	{
	  vm_data data2=pop_stack();
	  push_stack(data1);
	  push_stack(data2);
	}
    }
}

static void vm_i_plus()
{
  if (test_pop_stack("+"))
    {
      vm_data data1=pop_stack();
      if (test_pop_stack("+"))
	{
	  vm_data data2=pop_stack();
	  /* Check type of data */
	  if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_FLOAT))
	    {
	      if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_POINTER_TO_DATA))
		{
		  sprintf(comstring,"+ needs two floats or a variable and a float on stack\n");
		  vmputs(comstring);
		  vm_stop();
		}
	      else
		{
		  data2.data.myvarp.offset+=(unsigned int)(data1.data.mydouble+0.5);
		  push_stack(data2);
		}
	    }
	  else
	    {
	      data1.data.mydouble+=data2.data.mydouble;
	      /* No need to check whether it is ok to write to stack! */
	      push_stack(data1);
	    }
	}
    }
}

static void vm_i_minus()
{
  if (test_pop_stack("-"))
    {
      vm_data data1=pop_stack();
      if (test_pop_stack("-"))
	{
	  vm_data data2=pop_stack();
	  /* Check type of data */
	  if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_FLOAT))
	    {
	      if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_POINTER_TO_DATA))
		{
		  sprintf(comstring,"- needs two floats or a variable and a float on stack\n");
		  vmputs(comstring);
		  vm_stop();
		}
	      else
		{
		  unsigned int offset=(int)(data1.data.mydouble+0.5);
		  if (data2.data.myvarp.offset<offset)
		    {
		      vmputs("- on a variable resulted in negative offset\n");
		      vm_stop();
		    }
		  else
		    {
		      data2.data.myvarp.offset-=offset;
		      push_stack(data2);
		    }
		}
	    }
	  else
	    {
	      data2.data.mydouble-=data1.data.mydouble;
	      /* No need to check whether it is ok to write to stack! */
	      push_stack(data2);
	    }
	}
    }
}

static void vm_i_mul()
{
  if (test_pop_stack("*"))
    {
      vm_data data1=pop_stack();
      if (test_pop_stack("*"))
	{
	  vm_data data2=pop_stack();
	  /* Check type of data */
	  if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"* needs two floats on stack\n");
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      data2.data.mydouble*=data1.data.mydouble;
	      /* No need to check whether it is ok to write to stack! */
	      push_stack(data2);
	    }
	}
    }
}

static void vm_i_div()
{
  if (test_pop_stack("/"))
    {
      vm_data data1=pop_stack();
      if (test_pop_stack("/"))
	{
	  vm_data data2=pop_stack();
	  /* Check type of data */
	  if ((data1.type!=DATA_FLOAT) || (data2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"/ needs two floats on stack\n");
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (data1.data.mydouble==0)
		{
		  sprintf(comstring,"Division by 0 in /\n");
		  vmputs(comstring);
		  vm_stop();
		}
	      else
		{
		  data2.data.mydouble/=data1.data.mydouble;
		  /* No need to check whether it is ok to write to stack! */
		  push_stack(data2);
		}
	    }
	}
    }
}

static void vm_i_emit()
{
  /*  printf("In emit\n"); */
  if (test_pop_stack("emit"))
    {
      vm_data data=pop_stack();
      /* Check type of data */
      if (data.type!=DATA_FLOAT)
	{
	  sprintf(comstring,"Emit needs float on stack\n");
	  vmputs(comstring);
	  vm_stop();
	}
      else
	{
	  int c=(int)(data.data.mydouble);
	  sprintf(comstring,"%c",c);
	  vmputs(comstring);
	}
    }
}

static void vm_i_point()
{
  if (test_pop_stack("."))
    {
      vm_data data=pop_stack();
      /* Check type of data */
      if ((data.type!=DATA_FLOAT) && (data.type!=DATA_POINTER_TO_DATA) && (data.type!=DATA_POINTER_TO_FUNCTION))
	{
	  sprintf(comstring,". needs float on stack\n");
	  vmputs(comstring);
	  vm_stop();
	}
      else
	{
	  if (data.type==DATA_FLOAT)
	    sprintf(comstring,"%f",data.data.mydouble);
	  else if (data.type==DATA_POINTER_TO_DATA)
	    sprintf(comstring,"%x:%x",data.data.myvarp.address,data.data.myvarp.offset);
	  else if (data.type==DATA_POINTER_TO_FUNCTION)
	    sprintf(comstring,"%x",data.data.myint);
	  vmputs(comstring);
	}
    }
}

static void vm_i_stop()
{
  vm_stop();
}

static void vm_i_nop()
{
}

static void vm_i_mstat()
{
  sprintf(comstring,"pc=%d, sp=%d, rsp=%d\n",vm_pc,vm_sp,vm_rsp);
  vmputs(comstring);
}

static void vm_i_cdump()
{
  char t[1000];
  unsigned int i=0;
  unsigned int clen=0;
  strcpy(comstring,"     0: ");
  while (i<vm_code_len)
    {
      sprintf(t,"*%3u;%10u ",(unsigned int) vm_code_segment[i].type,vm_code_segment[i].data.myint);
      if (clen>=4)
	{
	  strcat(comstring,"\n");
	  vmputs(comstring);
	  sprintf(comstring,"%6u: ",i);
	  clen=0;
	}
      strcat(comstring,t);
      clen++;
      i++;
    }
  if (clen>0)
    {
      strcat(comstring,"\n");
      vmputs(comstring);
    }
}

static void vm_i_stat()
{
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"Y4 virtual machine\n");
  vmputs(comstring);
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"memory stat (units of y4 code words and y4 data words)\n");
  vmputs(comstring);
  sprintf(comstring,"y4 code word size=%d bytes\n",(int)sizeof(vm_instr));
  vmputs(comstring);
  sprintf(comstring,"y4 data word size=%d bytes\n",(int)sizeof(vm_data));
  vmputs(comstring);
  sprintf(comstring,"          code     stack    rstack      data     dynamic\n");
  vmputs(comstring);
  sprintf(comstring,"Size    %6d    %6d    %6d    %6d    %8d (in %d blocks)\n",vm_code_size,vm_stack_size,vm_rstack_size,vm_data_size,vm_dynamic_memory_amount,vm_dynamic_memory_blocks);
  vmputs(comstring);
  sprintf(comstring,"Used    %6d    %6d    %6d    %6d\n",vm_code_len,vm_sp,vm_rsp,vm_data_len);
  vmputs(comstring);
  sprintf(comstring,"Free    %6d    %6d    %6d    %6d\n",vm_code_size-vm_code_len,vm_stack_size-vm_sp,vm_rstack_size-vm_rsp,vm_data_size-vm_data_len);
  vmputs(comstring);
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
  sprintf(comstring,"Virtual machine clock ticks: %d\n",nticks);
  vmputs(comstring);
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
}

static void vm_i_list()
{
  i_table *i=my_table;
  f_table *f=my_f_table;
  e_table *e=my_e_table;
  var_table *v=my_var_table;
  mywords_t *t=mywords;
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"Y4 word list\n");
  vmputs(comstring);
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
  sprintf(comstring,"Internal words\n");
  vmputs(comstring);
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"Name\n");
  vmputs(comstring);
  /* Print some special words for user convenience */
  vmputs("if\n");
  vmputs("then\n");
  vmputs("else\n");
  vmputs("'\n");
  vmputs(":\n");
  vmputs(";\n");
  vmputs("(\n");
  vmputs(")\n");
  vmputs("var\n");
  while(i->name!=NULL)
    {
      sprintf(comstring,"%s\n",i->name);
      vmputs(comstring);
      i++;
    }
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"External words\n");
  vmputs(comstring);
  sprintf(comstring,"Handle      Name\n");
  vmputs(comstring);
  while(e!=NULL)
    {
      sprintf(comstring,"%6d      %s\n",e->handle,e->name);
      vmputs(comstring);
      e=e->next;
    }
  sprintf(comstring,"****************************************\n");
  vmputs(comstring);
  sprintf(comstring,"User declared words\n");
  vmputs(comstring);
  sprintf(comstring,"Address     Name\n");
  vmputs(comstring);
  while(f!=NULL)
    {
      sprintf(comstring," %6d     %s\n",f->f_pc,f->name);
      vmputs(comstring);
      f=f->next;
    }
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
  sprintf(comstring,"Variables\n");
  vmputs(comstring);
  sprintf(comstring,"Address       Size      Name\n");

  vmputs(comstring);
  while(v!=NULL)
    {
      sprintf(comstring," %6d     %6d      %s\n",v->address,v->size,v->name);
      vmputs(comstring);
      v=v->next;
    }
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
  sprintf(comstring,"Derived types\n");
  vmputs(comstring);
  sprintf(comstring,"Size      Name\n");
  vmputs(comstring);
  while(t!=NULL)
    {
      if (t->type==WORD_DERIVED_TYPE)
	{
	  sprintf(comstring," %6d      %s\n",t->gen->data.total_size,t->name);
	  vmputs(comstring);
	}
      t=t->next;
    }
  sprintf(comstring,"****************************************\n\n");
  vmputs(comstring);
}

static void vm_call(unsigned int f_pc)
{
  /*  printf("In vm_call\n"); */
  if (test_push_rstack("__call"))  
    {
      vm_data data;
      data.type=DATA_POINTER_TO_CODE;
      data.data.myint=vm_pc;
      push_rstack(data);
      /* This has to be -1 because vm_pc is increased later... */
      vm_pc=f_pc-1;
    }
}

static void vm_jump(unsigned int f_pc)
{
  /* This has to be -1 because vm_pc is increased later... */
  vm_pc=f_pc-1;
}

static void vm_i_ret()
{
  /*  printf("In vm_i_returnl\n"); */
  if (test_pop_rstack("__ret"))
    {
      vm_data data=pop_rstack();
      if (data.type!=DATA_POINTER_TO_CODE)
	{
	  sprintf(comstring,"Cannot return from function if no code pointer is on the return stack\n");
	  vmputs(comstring);
	  vm_stop();
	}
      else
	{
	  vm_pc=data.data.myint;
	}
    }
}

static void vm_i_do()
{
  if (test_pop_stack("do"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("do"))
	{
	  vm_data p2=pop_stack();
	  /* Check type of data */
	  if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"do needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (test_push_rstack("do"))
		{
		  vm_data addr;
		  addr.type=DATA_POINTER_TO_CODE;
		  /* Point to next instruction */
		  addr.data.myint=vm_pc+1;
		  push_rstack(addr);
		  if (test_push_rstack("do"))
		    {
		      push_rstack(p2);
		      if (test_push_rstack("do"))
			{
			  push_rstack(p1);
			}
		    }
		}
	    }
	}
    }
}

static void vm_i_loop()
{
  if (test_pop_rstack("loop"))
    {
      vm_data p1=pop_rstack();
      if (test_pop_rstack("loop"))
	{
	  vm_data p2=pop_rstack();
	  if (test_pop_rstack("loop"))
	    {
	      vm_data p3=pop_rstack();
	      if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT) || (p3.type!=DATA_POINTER_TO_CODE))
		{
		  sprintf(comstring,"loop needs two floats and one address on return stack \n");	      
		  vmputs(comstring);
		  vm_stop();
		}
	      else
		{
		  p1.data.mydouble++;
		  if (p1.data.mydouble<=p2.data.mydouble)
		    {
		      /* Continue loop */
		      /* No need to test for space on return stack */
		      push_rstack(p3);
		      push_rstack(p2);
		      push_rstack(p1);
		      vm_jump(p3.data.myint);
		    }
		}
	    }
	}      
    }
}

static void vm_i_begin()
{
  if (test_push_rstack("begin"))
    {
      vm_data addr;
      addr.type=DATA_POINTER_TO_CODE;
      /* Point to next instruction */
      addr.data.myint=vm_pc+1;
      push_rstack(addr);
    }
}

static void vm_i_until()
{
  if (test_pop_stack("until"))
    {
      vm_data test=pop_stack();
      if (test_pop_rstack("until"))
	{
	  vm_data addr=pop_rstack();
	  if ((test.type!=DATA_FLOAT) || (addr.type!=DATA_POINTER_TO_CODE))
	    {
	      vmputs("until needs a float on stack and an address on the return stack");
	      vm_stop();
	    }
	  else
	    {
	      if (test.data.mydouble==0)
		{
		  /* Continue loop */
		  /* No need to test for space on return stack */
		  push_rstack(addr);
		  vm_jump(addr.data.myint);
		}
	    }
	}
    }
}

static void vm_i_r()
{
  /*  vmputs("In r\n"); */
  if (test_pop_rstack("r"))
    {
      vm_data data=pop_rstack();
      push_rstack(data);
      if (test_push_stack("r"))
	push_stack(data);
    }  
}

static void vm_i_g_r()
{
  /*  vmputs("In >r\n"); */
  if (test_pop_stack(">r"))
    {
      vm_data data=pop_stack();
      if (test_push_rstack(">r"))
	push_rstack(data);
    }  
}

static void vm_i_r_g()
{
  /*  vmputs("In >r\n"); */
  if (test_pop_rstack("r>"))
    {
      vm_data data=pop_rstack();
      if (test_push_stack("r>"))
	push_stack(data);
    }  
}

static void vm_i_gt()
{
  /*  vmputs("In >\n"); */
  if (test_pop_stack(">"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack(">"))
	{
	  vm_data p2=pop_stack();
	  if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"> needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (p2.data.mydouble>p1.data.mydouble)
		p1.data.mydouble=1;
	      else
		p1.data.mydouble=0;
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	}
    }  
}

static void vm_i_lt()
{
  /*  vmputs("In <\n"); */
  if (test_pop_stack("<"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("<"))
	{
	  vm_data p2=pop_stack();
	  if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"< needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (p2.data.mydouble<p1.data.mydouble)
		p1.data.mydouble=1;
	      else
		p1.data.mydouble=0;
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	}
    }  
}

static void vm_i_eq()
{
  if (test_pop_stack("="))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("="))
	{
	  vm_data p2=pop_stack();
	  if (((p1.type==DATA_FLOAT) && (p2.type==DATA_FLOAT))
	      || (((p1.type==DATA_POINTER_TO_FUNCTION)
		  || (p1.type==DATA_POINTER_TO_DATA))
		  && ((p2.type==DATA_POINTER_TO_FUNCTION)
		  || (p2.type==DATA_POINTER_TO_DATA))))
	    {
	      if (p1.type==DATA_FLOAT)
		{
		  if (p1.data.mydouble==p2.data.mydouble)
		    p1.data.mydouble=1;
		  else
		    p1.data.mydouble=0;
		}
	      else
		{
		  int c1,c2;
		  if (p1.type==DATA_POINTER_TO_FUNCTION)
		    c1=p1.data.myint;
		  else
		    c1=p1.data.myvarp.address;		    
		  if (p2.type==DATA_POINTER_TO_FUNCTION)
		    c2=p2.data.myint;
		  else
		    c2=p2.data.myvarp.address;
		  if (c1==c2)
		    p1.data.mydouble=1;
		  else
		    p1.data.mydouble=0;
		  p1.type=DATA_FLOAT;
		}
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	  else
	    {
	      sprintf(comstring,"= needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	}
    }  
}

static void vm_i_ne()
{
  if (test_pop_stack("!="))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("!="))
	{
	  vm_data p2=pop_stack();
	  if (((p1.type==DATA_FLOAT) && (p2.type==DATA_FLOAT))
	      || (((p1.type==DATA_POINTER_TO_FUNCTION)
		  || (p1.type==DATA_POINTER_TO_DATA))
		  && ((p2.type==DATA_POINTER_TO_FUNCTION)
		  || (p2.type==DATA_POINTER_TO_DATA))))
	    {
	      if (p1.type==DATA_FLOAT)
		{
		  if (p1.data.mydouble!=p2.data.mydouble)
		    p1.data.mydouble=1;
		  else
		    p1.data.mydouble=0;
		}
	      else
		{
		  int c1,c2;
		  if (p1.type==DATA_POINTER_TO_FUNCTION)
		    c1=p1.data.myint;
		  else
		    c1=p1.data.myvarp.address;		    
		  if (p2.type==DATA_POINTER_TO_FUNCTION)
		    c2=p2.data.myint;
		  else
		    c2=p2.data.myvarp.address;
		  if (c1!=c2)
		    p1.data.mydouble=1;
		  else
		    p1.data.mydouble=0;
		  p1.type=DATA_FLOAT;
		}
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	  else
	    {
	      sprintf(comstring,"!= needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	}
    }  
}

static void vm_i_gte()
{
  if (test_pop_stack(">="))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack(">="))
	{
	  vm_data p2=pop_stack();
	  if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"> needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (p2.data.mydouble>=p1.data.mydouble)
		p1.data.mydouble=1;
	      else
		p1.data.mydouble=0;
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	}
    }  
}

static void vm_i_lte()
{
  if (test_pop_stack("<="))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("<="))
	{
	  vm_data p2=pop_stack();
	  if ((p1.type!=DATA_FLOAT) || (p2.type!=DATA_FLOAT))
	    {
	      sprintf(comstring,"> needs two floats on stack\n");	      
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      if (p2.data.mydouble<=p1.data.mydouble)
		p1.data.mydouble=1;
	      else
		p1.data.mydouble=0;
	      /* No need to test for space on stack */
	      push_stack(p1);
	    }
	}
    }  
}

static void vm_if(unsigned int f_pc)
{
  if (test_pop_stack("if"))
    {
      vm_data p1=pop_stack();
      if (p1.type!=DATA_FLOAT)
	{
	  sprintf(comstring,"if needs float on stack\n");
	  vmputs(comstring);
	  vm_stop();
	}
      else
	{
	  if (p1.data.mydouble==0)
	    {
	      /* Skip instruction */
	      vm_jump(f_pc);
	    }
	}
    }
}

static void vm_else(unsigned int f_pc)
{
  /* Skip instruction */
  vm_jump(f_pc);
}

char *vm_get_string_from_vm(vm_data_type type, unsigned int ptr)
{
  /* Check string length */
  char *mystr=NULL;
  unsigned int tmp=ptr;
  unsigned int test;
  if (type==DATA_POINTER_TO_STRING)
    {
      test=vm_data_size;
      while ((tmp<test) &&
	     (vm_address(tmp)->data.myint!=0))
	tmp++;
    }
  else
    {
      test=vm_code_size;
    while ((tmp<test) &&
	   (vm_code_segment[tmp].data.myint!=0))
      tmp++;
    }

  if (tmp>=test)
    {
      vmputs("Fatal: String doesn't end\n");
      vm_stop();
    }
  else
    {
      {
	int i=0;
	mystr=malloc((tmp-ptr+1));
	if (type==DATA_POINTER_TO_STRING)
	  while (vm_address(ptr)->data.myint!=0)
	    mystr[i++]=(char)(vm_address(ptr++)->data.myint);
	else
	  while (vm_code_segment[ptr].data.myint!=0)
	    mystr[i++]=(char)(vm_code_segment[ptr++].data.myint);

	mystr[i++]=0;
      }
    }
  return(mystr);
}

static void vm_i_printf()
{
  if (test_pop_stack("printf"))
    {
      vm_data p1=pop_stack();
      if ((p1.type!=DATA_POINTER_TO_STRING) && (p1.type!=DATA_POINTER_TO_CODE_STRING))
	{
	  sprintf(comstring,"printf needs string on stack\n");
	  vmputs(comstring);
	  vm_stop();
	}
      else
	{
	  char *cstring=vm_get_string_from_vm(p1.type,p1.data.myint);
	  if (cstring!=NULL)
	    {
	      vmputs(cstring);
	      free(cstring);
	    }
	}
    }
}

static void vm_i_load()
{
  if (test_pop_stack("@"))
    {
      vm_data p1=pop_stack();
      if (p1.type!=DATA_POINTER_TO_DATA)
	{
	  vmputs("@ needs a variable on stack\n");
	  vm_stop();
	}
      else
	{
	  if ((p1.data.myvarp.offset>=0) && (p1.data.myvarp.offset<p1.data.myvarp.size))
	    {
	      vm_data data=*vm_address(p1.data.myvarp.address+p1.data.myvarp.offset);
	      push_stack(data);
	    }
	  else
	    {
	      vmputs("Error: Array bounds in @\n");
	      sprintf(comstring,"Offset is %d but it is required that 0<=offset<=%d.\n",
		      p1.data.myvarp.offset,p1.data.myvarp.size-1);
	      vmputs(comstring);
	      vm_stop();
	    }
	}
    }
}

static void vm_i_store()
{
  if (test_pop_stack("!"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("!"))
	{
	  vm_data p2=pop_stack();
	  if (p1.type!=DATA_POINTER_TO_DATA)
	    {
	      vmputs("! needs a variable and some data on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      if ((p1.data.myvarp.offset>=0) && (p1.data.myvarp.offset<p1.data.myvarp.size))
		{
		  *vm_address(p1.data.myvarp.address+p1.data.myvarp.offset)=p2;
		}
	      else
		{
		  vmputs("Error: Array bounds in !\n");
		  sprintf(comstring,"Offset is %d but it is required that 0<=offset<=%d.\n",
			  p1.data.myvarp.offset,p1.data.myvarp.size-1);
		  vmputs(comstring);
		  vm_stop();
		}
	    }
	}
    }
}

static void vm_i_leave()
{
  if (test_pop_rstack("__leave"))
    {
      vm_data p1=pop_rstack();
      if (test_pop_rstack("__leave"))
	{
	  vm_data p2=pop_rstack();
	  if (test_pop_rstack("__leave"))
	    {
	      vm_data p3=pop_rstack();
	      if ((p1.type!=DATA_FUNCTION_LEVEL) || (p2.type!=DATA_LOCAL_STORAGE) || (p3.type!=DATA_VM_DATA_LEN))
		{
		  vmputs("Fatal: Corrupted data on stack. Cannot return from function\n");
		  vm_stop();
		}
	      else
		{
		  vm_local_storage[p1.data.myint-1]=p2.data.myint;
		  vm_data_len=p3.data.myint;
		}
	    }
	}
    }
}

#if 0
static void vm_i_forget()
{
  if (test_pop_stack("forget"))
    {
      vm_data p1=pop_stack();
      if ((p1.type!=DATA_POINTER_TO_DATA) && (p1.type!=DATA_POINTER_TO_FUNCTION))
	{
	  vmputs("forget needs a variable or an address to a function on stack\n");
	  vm_stop();
	}
      else
	{
	  if (p1.type==DATA_POINTER_TO_DATA)
	    {
	      /* Check that user is really forgetting the last allocated variable */
	      if (my_var_table!=NULL)
		{
		  if (my_var_table->address!=p1.data.myint)
		    {
		      vmputs("Fatal: tried to forget a variable other than the last declared one\n");
		      vm_stop();
		    }
		  else
		    {
		      var_table *tmp=my_var_table->next;
		      free(my_var_table->name);
		      vm_data_len=my_var_table->address;
		      free(my_var_table);
		      my_var_table=tmp;
		    }
		}
	    }
	  else /* Pointer to function */
	    {
	      /* Forget all functions including this one after this one */
	      forget_function(p1.data.myint);
	    }
	}
    }
}
#endif

static void vm_i_cast()
{
  if (test_pop_stack("cast"))
    {
      vm_data data=pop_stack();
      if (data.type!=DATA_CAST)
	{
	  vmputs("cast needs type on stack\n");
	  vm_stop();
	}
      else
	{
	  unsigned int type=data.data.myint;
	  if (test_pop_stack("cast"))
	    {
	      data=pop_stack();
	      if (data.type!=DATA_POINTER_TO_DATA)
		{
		  vmputs("cast needs variable on stack\n");
		  vm_stop();
		}
	      else
		{
		  switch(type)
		    {
		    case CAST_STRING:
		      data.type=DATA_POINTER_TO_STRING;
		      break;
		    case CAST_INTEGER:
		      data.type=DATA_POINTER_TO_INTEGER;
		      break;
		    }
		  push_stack(data);
		}
	    }
	}
    }
}

static void vm_i_string()
{
  if (test_push_stack("string"))
    {
      vm_data data;
      data.type=DATA_CAST;
      data.data.myint=CAST_STRING;
      push_stack(data);
    }
}

static void vm_i_strcpy()
{
  if (test_pop_stack("strcpy"))
    {
      vm_data data=pop_stack();
      if (test_pop_stack("strcpy"))
	{
	  vm_data data2=pop_stack();
	  if (data.type!=DATA_POINTER_TO_STRING)
	    {
	      vmputs("strcpy needs string variable on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      if ((data2.type!=DATA_POINTER_TO_STRING) && (data2.type!=DATA_POINTER_TO_CODE_STRING))
		{
		  vmputs("strcpy needs strings/string variables on stack\n");
		  vm_stop();
		}
	      else
		{
		  /* Get string */
		  char *mystr=vm_get_string_from_vm(data2.type,data2.data.myint);
		  int i=strlen(mystr);
		  /* Test length */
		  if (i>=data.data.myvarp.size-data.data.myvarp.offset)
		    {
		      vmputs("String too long in strcpy\n");
		      vm_stop();
		    }
		  else
		    {
		      int start;
		      int i_ptr=data.data.myvarp.offset+data.data.myvarp.address;
		      /* printf("Copying to %d\n",i_ptr); */
		      start=0;
		      while ((i=mystr[start++])!=0)
			vm_address(i_ptr++)->data.myint=i;
		      vm_address(i_ptr)->data.myint=0;
		    }
		}
	    }
	}
    }
}

static void vm_i_itos()
{
  if (test_pop_stack("strcpy"))
    {
      vm_data data=pop_stack();
      if (test_pop_stack("strcpy"))
	{
	  vm_data data2=pop_stack();
	  if (data.type!=DATA_POINTER_TO_STRING)
	    {
	      vmputs("itos needs string variable on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      if (data2.type!=DATA_FLOAT)
		{
		  vmputs("itos needs float on stack\n");
		  vm_stop();
		}
	      else
		{
		  /* Get string */
		  char mystr[100];
		  int i=(int)data2.data.mydouble;
		  sprintf(mystr,"%d",i);
		  i=strlen(mystr);
		  /* Test length */
		  if (i>=data.data.myvarp.size-data.data.myvarp.offset)
		    {
		      vmputs("String too short to hold number in itos\n");
		      vm_stop();
		    }
		  else
		    {
		      int start;
		      int i_ptr=data.data.myvarp.offset+data.data.myvarp.address;
		      /* printf("Copying to %d\n",i_ptr); */
		      start=0;
		      while ((i=mystr[start++])!=0)
			vm_address(i_ptr++)->data.myint=i;
		      vm_address(i_ptr)->data.myint=0;
		    }
		}
	    }
	}
    }
}

static void vm_i_strcat()
{
  if (test_pop_stack("strcat"))
    {
      vm_data data=pop_stack();
      if (test_pop_stack("strcat"))
	{
	  vm_data data2=pop_stack();
	  if (data.type!=DATA_POINTER_TO_STRING)
	    {
	      vmputs("strcat needs string variable on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      if ((data2.type!=DATA_POINTER_TO_STRING) && (data2.type!=DATA_POINTER_TO_CODE_STRING))
		{
		  vmputs("strcat needs strings/string variables on stack\n");
		  vm_stop();
		}
	      else
		{
		  /* Get strings */
		  char *mystr=vm_get_string_from_vm(data2.type,data2.data.myint);
		  int i=strlen(mystr);
		  char *mystr2=vm_get_string_from_vm(data.type,data.data.myint);
		  int i2=strlen(mystr2);
		  
		  /* Test length */
		  if ((i+i2)>=data.data.myvarp.size-data.data.myvarp.offset)
		    {
		      vmputs("String too long in strcat\n");
		      vm_stop();
		    }
		  else
		    {
		      int start;
		      int i_ptr=data.data.myvarp.offset+data.data.myvarp.address+i2;
		      start=0;
		      while ((i=mystr[start++])!=0)
			vm_address(i_ptr++)->data.myint=i;
		      vm_address(i_ptr)->data.myint=0;
		    }
		}
	    }
	}
    }
}

static void vm_sw_interrupt(int nr)
{
  /* printf("SW=%d\n",nr); */
  if ((nr>=0) && (nr<MAX_INTERRUPTS))
    {
      vm_interrupt *ptr=int_vector[nr];
      while (ptr!=NULL)
	{
	  if (test_push_rstack("__int"))
	    {
	      vm_data data;
	      data.type=DATA_FLAG;
	      data.data.myint=interrupt_flag;
	      push_rstack(data);
	      interrupt_flag=0; /* CLI */
	      if (test_push_rstack("__int"))
		{
		  vm_data data;
		  data.type=DATA_POINTER_TO_CODE;
		  data.data.myint=vm_pc;
		  push_rstack(data);
		  /* printf("SW2\n"); */
		  /* This has to be -1 because vm_pc is increased later... */
		  vm_pc=ptr->addr-1;
		  /* printf("SW pc=%d\n",vm_pc);  */
		}
	    }
	  ptr=ptr->next;
	}
    }
}

static void vm_i_iret()
{
  if (test_pop_rstack("__iret"))
    {
      vm_data data=pop_rstack();
      if (test_pop_rstack("__iret"))
	{
	  vm_data data2=pop_rstack();
	  if ((data.type!=DATA_POINTER_TO_CODE) || (data2.type!=DATA_FLAG))
	    {
	      sprintf(comstring,"Cannot iret from function\n");
	      vmputs(comstring);
	      vm_stop();
	    }
	  else
	    {
	      vm_pc=data.data.myint;
	      /* printf("__iret before: if=%d\n",interrupt_flag); */
	      interrupt_flag=data2.data.myint;
	      /* printf("__iret after: if=%d\n",interrupt_flag); */
	    }
	}
    }
}

static int execute_hw_interrupt=0;
static int execute_hw_interrupt_nr;

static void delayed_vm_hw_interrupt()
{
  /* printf("HW pc=%d\n",vm_pc); */
    vm_pc--;
    vm_sw_interrupt(execute_hw_interrupt_nr);
    vm_pc++;
    execute_hw_interrupt=0;
}

void vm_hw_interrupt(int nr)
{
  /* printf("if=%d\n",interrupt_flag); */
  if (interrupt_flag)
    {
      if (execute_hw_interrupt)
	{
	  vmputs("Warning: Lost interrupt\n");
	}
      execute_hw_interrupt=1;
      execute_hw_interrupt_nr=nr;
    }
}

static void vm_i_int()
{
  if (test_pop_stack("__int"))
    {
      vm_data p1=pop_stack();
      if (p1.type!=DATA_FLOAT)
	{
	  vmputs("__int needs a float on stack\n");
	  vm_stop();
	}
      else
	{
	  vm_sw_interrupt((int)p1.data.mydouble);
	}
    }
}


static void vm_i_call()
{
  if (test_pop_stack("__call"))
    {
      vm_data p1=pop_stack();
      if ((p1.type!=DATA_FLOAT) && (p1.type!=DATA_POINTER_TO_FUNCTION))
	{
	  vmputs("__call needs a float or a function pointer on stack\n");
	  vm_stop();
	}
      else
	{
	  if (p1.type==DATA_FLOAT)
	    vm_call((int)(p1.data.mydouble+0.1));
	  else
	    vm_call(p1.data.myint);
	}
    }
}

static void vm_i_install()
{
  if (test_pop_stack("install"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("install"))
	{
	  vm_data p2=pop_stack();
	  if ((p2.type!=DATA_POINTER_TO_FUNCTION) || (p1.type!=DATA_FLOAT))
	    {
	      vmputs("install needs an interrupt nr and pointer to a function on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      add_interrupt((int)p1.data.mydouble,p2.data.myint);
	    }
	}
    }
}

static void vm_i_deinstall()
{
  if (test_pop_stack("uninstall"))
    {
      vm_data p1=pop_stack();
      if (test_pop_stack("uninstall"))
	{
	  vm_data p2=pop_stack();
	  if ((p2.type!=DATA_POINTER_TO_FUNCTION) || (p1.type!=DATA_FLOAT))
	    {
	      vmputs("deinstall needs an interrupt nr and pointer to a function on stack\n");
	      vm_stop();
	    }
	  else
	    {
	      remove_interrupt((int)p1.data.mydouble,p2.data.myint);
	    }
	}
    }
}

static void vm_i_null()
{
  if (test_push_stack("NULL"))
    {
      vm_data d;
      d.type=DATA_POINTER_TO_FUNCTION;
      d.data.myint=0;
      push_stack(d);
    }  
}

/* Array of internal instruction function pointers */
static void (*myintiarr[MAX_INTERNALS])()=
{
  vm_i_dup,
  vm_i_drop,
  vm_i_plus,
  vm_i_minus,
  vm_i_emit,
  vm_i_stop,
  vm_i_nop,
  vm_i_stat,
  vm_i_list,
  vm_i_ret,
  vm_i_mul,
  vm_i_div,
  vm_i_point,
  vm_i_do,
  vm_i_loop,
  vm_i_r,
  vm_i_g_r,
  vm_i_r_g,
  vm_i_gt,
  vm_i_lt,
  vm_i_eq,
  vm_i_gte,
  vm_i_gte,
  vm_i_lte,
  vm_i_lte,
  vm_i_ne,
  vm_i_printf,
  vm_i_begin,
  vm_i_until,
  vm_i_swap,
  vm_i_load,
  vm_i_store,
  vm_i_leave,
  vm_i_mstat,
  vm_i_cdump,
  vm_i_cast,
  vm_i_string,
  vm_i_strcpy,
  vm_i_int,
  vm_i_iret,
  vm_i_install,
  vm_i_deinstall,
  vm_i__stop,
  vm_i_itos,
  vm_i_strcat,
  vm_i_call,
  vm_i_malloc,
  vm_i_free,
  vm_i_null,
};

int vm_tick()
{
  if (vm_running)
    {
      vm_instr *myinstr=vm_code_segment+vm_pc;
      /* printf("exec pc=%d\n",vm_pc); */
      nticks++;
      switch(myinstr->type)
	{
	case INSTR_INTERNAL:
	  /* sprintf(comstring,"Internal instr: %d\n",myinstr->data.myint);
	  vmputs(comstring); */
	  myintiarr[myinstr->data.myint]();
	  break;
	case INSTR_ENTER:
	  {
	    if (test_push_rstack("__enter"))
	      {
		vm_data data;
		data.type=DATA_VM_DATA_LEN;
		data.data.myint=vm_data_len;
		push_rstack(data);
		if (test_push_rstack("__enter"))
		  {
		    data.type=DATA_LOCAL_STORAGE;
		    data.data.myint=vm_local_storage[myinstr->data.myint-1];
		    push_rstack(data);
		    vm_local_storage[myinstr->data.myint-1]=vm_data_len;
		    if (test_push_rstack("__enter"))
		      {
			data.type=DATA_FUNCTION_LEVEL;
			data.data.myint=myinstr->data.myint;
			push_rstack(data);
		      }
		  }
	      }
	  }
          break;
	case INSTR_FUNCTION_CALL:
	  vm_call(myinstr->data.myint);
	  break;
	case INSTR_VARALLOT:
	  if ((vm_data_len+myinstr->data.myint)>vm_data_size)
	    {
	      vmputs("Out of data memory on variable allocation\n");
	      vm_stop();
	    }
	  else
	    vm_data_len+=myinstr->data.myint;
	  break;
	case INSTR_VARIABLE:
	  {
	    if (test_push_stack("variable"))
	      {
		vm_data data;
		data.type=DATA_POINTER_TO_DATA;
		data.data.myvarp=myinstr->data.myvarp;
		if (data.data.myvarp.local)
		  {
		    data.data.myvarp.address+=vm_local_storage[data.data.myvarp.local-1];
		    /* sprintf(comstring,"vm_local_storage=%d, address=%d\n",vm_local_storage,
			    data.data.myvarp.address);
		    vmputs(comstring); */
		  }
		push_stack(data);
	      }
	  }
	break;
	case INSTR_EXTERNAL:
	  call_external_function(myinstr->data.myint);
	  break;
	case INSTR_NEW_EXTERNAL:
	  if (valid_machine)
	    {
	      void (*x)(void);
	      x=myinstr->data.myptr;
 	      x();
	    }
	  break;
	case INSTR_JUMP:
	  vm_jump(myinstr->data.myint);
	  break;
	case INSTR_JUMP_ONCE:
	  {
	    unsigned int jumpto=myinstr->data.myint;
	    /* Change this jump to a NOP */
	    myinstr->type=INSTR_INTERNAL;
	    myinstr->data.myint=I_NOP;
	    vm_jump(jumpto);
	  }
	  break;
	case INSTR_IF:
	  vm_if(myinstr->data.myint);
	  break;
	case INSTR_ELSE:
	  vm_else(myinstr->data.myint);
	  break;
	case INSTR_THEN:
	  break;
	case INSTR_FLOAT:
	  if (test_push_stack("float"))
	    {
	      vm_data data;
	      data.type=DATA_FLOAT;
	      data.data.mydouble=myinstr->data.mydouble;
	      push_stack(data);
	    }
	  break;
	case INSTR_POINTER_TO_STRING:
	  if (test_push_stack("string"))
	    {
	      /*Push pointer to string on stack */
	      vm_data data;
	      unsigned int i=0;
	      unsigned int start=myinstr->data.myint;
	      data.type=DATA_POINTER_TO_CODE_STRING;
	      data.data.myint=myinstr->data.myint;
	      push_stack(data);
	    }
	  break;
	case INSTR_POINTER_TO_FUNCTION:
	  if (test_push_stack("$"))
	    {
	      /*Push pointer to function on stack */
	      vm_data data;
	      data.type=DATA_POINTER_TO_FUNCTION;
	      data.data.myint=myinstr->data.myint;
	      push_stack(data);
	    }
	  break;
	case INSTR_DATA:
	  sprintf(comstring,"Fatal: Ran into data in code segment\n");
	  vmputs(comstring);
	  vm_stop();
	  break;
	};
  
      vm_pc++;
      if (vm_pc>=vm_code_size)
	{
	  sprintf(comstring,"Reached end of code segment\n");
	  vmputs(comstring);
	  vm_stop();
	}
    }
  if (execute_hw_interrupt)
    delayed_vm_hw_interrupt();

  return vm_running;
}

char *vm_get_version_string()
{
  return(PROGRAMVERSION);
}

void fiddle_with_pc(unsigned int i)
{
  vm_pc+=i;
}

