/* This is vmconsole.c
   A part of the Ymol program
   Copyright (C) 1997-1998 Daniel Spangberg
   */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>

#include <Xco.h>
#include <y4vm.h>
#include <y4vmmath.h>
#include <y4vmload.h>
#include <y4vmfile.h>
#include "userinterface.h"
#include "ymolvm.h"
#include "ymolrc.h"


#define NTICKSPERFRAME 50000
#define MAXLINES 100
#define MAXCPL 80
#define MYFONT "fixed"

#define COLOR1 PIXEL(0,0,0)
#define COLOR2 PIXEL(255,255,255)

static int vm_is_running=0;
static int inosync=1;

static void display_cursor();
static void remove_cursor();

void vm_sync()
{
  inosync=0;
}

void vm_synced()
{
  inosync=1;
}

static int vm_update()
{
  if (vm_is_running)
    {
      int i;

      remove_cursor();
      for(i=0; i<NTICKSPERFRAME; i++)
	{
	  if ((inosync) && (vm_is_running))
	    {
	      vm_is_running=vm_tick();
	    }
	}
      if (!vm_is_running)
	restore_vm_state();
    }
  else
    {
      display_cursor();
    }
  return(0);
}

void vmconsole_interrupt(int nr)
{
  if (!vm_is_running)
    {
      char tt[10];
      vm_run();
      vm_is_running=1;
      sprintf(tt,"%d __int",nr);
      vm_parse_and_compile(tt);
      vm_parse_and_compile("__stop");
      save_vm_state();
    }
  else
    {
      vm_hw_interrupt(nr);
    }
}

void vmconsole_call(int address,char *programtext)
{
  if (!vm_is_running)
    {
      char tt[100];
      vm_run();
      vm_is_running=1;
      vm_parse_and_compile(programtext);
      sprintf(tt,"%d __call",address);
      vm_parse_and_compile(tt);
      vm_parse_and_compile("__stop");
      save_vm_state();
    }
}

static XcoObject console,displaybox;
static int console_open=0;

static char mybuffer[MAXLINES][MAXCPL+1];
static int current_lines=1;

static XFontStruct *myfont;
static int fontheight;

#define CURSORCOUNT 20
static int cursor_status=0;
static int cursor_count=0;

static void remove_cursor()
{
  if (console_open)
    {
      int objheight=XcoGetObjectHeight(displaybox);
      int current_y=objheight-fontheight;
      int cxpos=XcoTextWidth(myfont,mybuffer[current_lines-1]);
      XSetForeground(XcoGetDisplay(),XcoGC(displaybox),XcoGetPixel32(0,0,COLOR2));
      XDrawLine(XcoGetDisplay(),XcoWindow(displaybox),XcoGC(displaybox),
		cxpos,current_y,cxpos,current_y+fontheight);
      cursor_status=0;
      cursor_count=0;
    }
}

static void my_display_cursor()
{
  if (console_open)
    {
      int objheight=XcoGetObjectHeight(displaybox);
      int current_y=objheight-fontheight;
      int cxpos=XcoTextWidth(myfont,mybuffer[current_lines-1]);
      XSetForeground(XcoGetDisplay(),XcoGC(displaybox),XcoGetPixel32(0,0,COLOR1));
      XDrawLine(XcoGetDisplay(),XcoWindow(displaybox),XcoGC(displaybox),
		cxpos,current_y,cxpos,current_y+fontheight);
      cursor_status=1;
      cursor_count=0;
    }
}

static void display_cursor()
{
  if (console_open)
    {
      cursor_count++;
      if (cursor_count>CURSORCOUNT)
	{
	  if (cursor_status)
	    remove_cursor();
	  else
	    my_display_cursor();
	}
    }
}

static void expose_line(int current_y,int line)
{
  XcoDrawFont(displaybox,0,current_y,mybuffer[line],COLOR1);
}

static void expose_clear_line(int current_y)
{
  int objwidth=XcoGetObjectWidth(displaybox);
  XSetForeground(XcoGetDisplay(),XcoGC(displaybox),XcoGetPixel32(0,0,COLOR2));
  XFillRectangle(XcoGetDisplay(),XcoWindow(displaybox),XcoGC(displaybox),
		 0,current_y,objwidth,fontheight);
}

static void expose_buffer(XcoObject id,XEvent event)
{
  if (event.type==Expose)
    {
      int i=current_lines;
      int objheight=XcoGetObjectHeight(displaybox);
      int current_y=objheight-fontheight;
      while(((current_y+fontheight)>0) && (i>=0))
	{
	  expose_line(current_y,i-1);
	  current_y-=fontheight;
	  i--;
	}
    }
}

static int linelen;
static void newline()
{
  int objheight=XcoGetObjectHeight(displaybox);
  int objwidth=XcoGetObjectWidth(displaybox);
  if (console_open)
    {
      remove_cursor();
      XCopyArea(XcoGetDisplay(),
		XcoWindow(displaybox),XcoWindow(displaybox),
		XcoGC(displaybox),
		0,fontheight,
		objwidth,objheight-fontheight,
		0,0);
      XSetForeground(XcoGetDisplay(),XcoGC(displaybox),XcoGetPixel32(0,0,COLOR2));
      XFillRectangle(XcoGetDisplay(),XcoWindow(displaybox),XcoGC(displaybox),
		     0,objheight-fontheight,objwidth,fontheight);
      /*
      XcoDrawFont(displaybox,0,objheight-fontheight,mybuffer[current_lines-1],COLOR1);
      */
      expose_line(objheight-fontheight-fontheight,current_lines-1);
      /* printf("%s\n",mybuffer[current_lines-1]); */
    }
  if (current_lines>=MAXLINES)
    {
      int i;
      for (i=0; i<(MAXLINES-1); i++)
	strcpy(mybuffer[i],mybuffer[i+1]);
    }
  else
    current_lines++;
  mybuffer[current_lines-1][0]='\0';
  linelen=0;
}

void display_line(char *str)
{
  char *myptr=str;
  char mychar;
  int objheight=XcoGetObjectHeight(displaybox);
  int objwidth=XcoGetObjectWidth(displaybox);
  int current_y=objheight-fontheight;
  linelen=strlen(mybuffer[current_lines-1]);
  remove_cursor();
  while((mychar=(*myptr++)))
    {
      if (mychar=='\n')
	{
	  mybuffer[current_lines-1][linelen]='\0';
	  newline();
	}
      else
	{
	  if (linelen>=MAXCPL)
	    {
	      mybuffer[current_lines-1][linelen]='\0';
	      newline();
	    }
	  mybuffer[current_lines-1][linelen]=mychar;
	  linelen++;
	}
    }
  mybuffer[current_lines-1][linelen]='\0';
  expose_line(current_y,current_lines-1);

  /* printf("%s",str); */
}

static void data_from_vm(char *str)
{
  if (console_open)
    display_line(str);
  else
    {
      printf(str);
      fflush(stdout);
    }
}

static int first_time=1;

void vmconsoleinit()
{
  char cstr[1000];
  if (first_time)
    vm_init();
  vm_clear();
  add_vm_send_data_handler(data_from_vm);
  vm_parse_and_compile("stop");
  save_vm_state();
  init_vmload();
  init_vmfile();
  /*  vmload_add_search_dir("./"); */
  init_vmmath();
  init_ymol_words();
  sprintf(cstr,"Y4vm console. Y4vm version %s\n",vm_get_version_string());
  vmputs(cstr);
  if (first_time)
    {
      XcoAddWorkProc(vm_update);
      myfont=XcoLoadFont(MYFONT);
      fontheight=XcoTextHeight(myfont);
      vm_run();
      vm_is_running=1;
    }
  first_time=0;
  RemoveToolMenu();
}

static void console_close()
{
  console_open=0;
  XcoDeleteObject(console);
}

static void cdelete(XcoObject id,XEvent event)
{
  if (XcoDeleteWindow(id,event))
    {
      console_close();
    }
}

static void console_dialog_callback(XcoObject id, XEvent event)
{
  if (event.type==KeyPress)
    {
      int length;
      char keyb_buffer[20];
      KeySym keysym;
      XComposeStatus composestatus;
      length=XLookupString(&event.xkey,keyb_buffer,1,
			   &keysym,&composestatus);
      keyb_buffer[length]=0;
      remove_cursor();
      switch(keysym)
	{
	case XK_BackSpace:
	  {
	    int len=strlen(mybuffer[current_lines-1]);
	    int objheight=XcoGetObjectHeight(displaybox);
	    int current_y=objheight-fontheight;
	    if (len>0)
	      mybuffer[current_lines-1][len-1]=0;
	    expose_clear_line(current_y);
	    expose_line(current_y,current_lines-1);
	  }
	  break;
	case XK_Return:
	  display_line("\n");
	  vm_run();
	  vm_is_running=1;
	  vm_parse_and_compile(mybuffer[current_lines-2]);
	  vm_parse_and_compile("stop");
	  save_vm_state();
	  break;
	default:
	  if (length!=0)
	    {
	      display_line(keyb_buffer);
	    }
	  break;
	}
    }
}

static void stop_callback(XcoObject id,XEvent event)
{
  if (event.type==ButtonPress)
    vm_stop();
}

static void reset_callback(XcoObject id,XEvent event)
{
  if (event.type==ButtonPress)
    {
      vmconsoleinit();
      save_vm_state();
      load_ymolrc();
      vm_invalidate();
    }
}


#define W_WIDTH 500
#define W_HEIGHT 300
#define W_KEYROW_HEIGHT 30

void open_vmconsole()
{
  if (!console_open)
    {
      XcoObject decbox,decbox2,decbox3,hole,oparent,resetbutton,stopbutton;
      console=XcoCreateNamedWindow(0,0,W_WIDTH,W_HEIGHT,DEFAULT_BACKGROUND,1,-1,"Y4 console");
      XSelectInput(XcoGetDisplay(),XcoWindow(console),
		   ExposureMask|ButtonPressMask|KeyPressMask|StructureNotifyMask);
      XcoAddCallback(console,cdelete);
      XcoAddCallback(console,console_dialog_callback);
      decbox2=XcoCreateBox3D(console,5,5,W_WIDTH-10,W_HEIGHT-10,2,1);
      decbox=XcoCreateBox3D(decbox2,5,5+W_KEYROW_HEIGHT,W_WIDTH-20,W_HEIGHT-20-W_KEYROW_HEIGHT,1,1);
      displaybox=XcoCreateHole(decbox,2,2,W_WIDTH-24,W_HEIGHT-24-W_KEYROW_HEIGHT);
      XcoSetBackground(displaybox,COLOR2);
      XcoSetResizeMethod(decbox,XcoUpLeftDownRight);
      XcoSetResizeMethod(displaybox,XcoUpLeftDownRight);
      hole=XcoCreateHole(decbox2,2,2,W_WIDTH-14,W_KEYROW_HEIGHT);
      decbox3=XcoCreateBox3D(hole,2,2,W_WIDTH-18,W_KEYROW_HEIGHT-4,2,1);
      XcoSetResizeMethod(hole,XcoUpResize);
      XcoSetResizeMethod(decbox3,XcoUpLeftDownRight);
      stopbutton=XcoCreateCommand(decbox3,5,4,"Stop",0,0);
      resetbutton=XcoCreateCommand(decbox3,15+XcoGetObjectWidth(stopbutton),4,"Reset",0,0);
      XcoSetResizeMethod(stopbutton,XcoUpLeft);
      XcoSetResizeMethod(resetbutton,XcoUpLeft);
      XcoAddCallback(stopbutton,stop_callback);
      XcoAddCallback(resetbutton,reset_callback);

      current_lines=1;
      mybuffer[0][0]='\0';
      XcoAddCallback(displaybox,expose_buffer);
      XcoUseFont(displaybox,myfont);
#if 0
      vm_parse_and_compile("stop");
      save_vm_state();
      vm_run();
      vm_is_running=1;
#endif
      console_open=1;
    }
  else
    {
      XRaiseWindow(XcoGetDisplay(),XcoWindow(console));
    }
}

