/* Open a INET socket at a default port, write the port number to a
   file.  "Poll" the socket using select. If data is available on the
   socket import it. Very restricted import is available for this
   option. */

#if defined(USE_BSDSOCK) || defined(USE_WINSOCK)

#ifndef USE_BSDSOCK
#define USE_BSDSOCK
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>
#ifdef USE_WINSOCK
#include <winsock.h>
#else /* WINSOCK */
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#define WE_ARE_ON_UNIX
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#define INVALID_SOCKET -1
#define SOCKET int
#define closesocket close
#endif /* WINSOCK */

#include <signal.h>
#include <errno.h>

#ifndef HAS_SOCKLEN_T
typedef int socklen_t;
#endif

#if (SIZEOF_LONG==4)
typedef unsigned long myuint32_t;
#elif (SIZEOF_INT==4)
typedef unsigned int myuint32_t;
#endif

#endif /* BSDSOCK or WINSOCK */

#include "ymolsock.h"

#ifdef USE_BSDSOCK

static int has_connection=0;
static int init_failure=1;
static int PORT=0;

#define COOKIE_LENGTH 16
static char magic_cookie[COOKIE_LENGTH+1];

#define MYMAXHOSTNAMELEN 256

static char ymol_hostname[MYMAXHOSTNAMELEN];
static struct hostent *ymol_host;
static myuint32_t ymol_address;

static SOCKET connsock=-1;

static void close_connection()
{
  if (has_connection)
    {
      closesocket(connsock);
      has_connection=0;
    }
}

static int gotsenderror=0;

static void senderroralarm(int signum)
{
  fflush(stdout);
  gotsenderror=1;
}

static int write_any_sock(void *msg,int len,SOCKET thesock)
{
    int glen,remain=len;
    char *this_msg=msg;
    struct sigaction old_action;
    struct sigaction new_action;
    sigset_t sigset;
    sigemptyset(&sigset);
    new_action.sa_handler=senderroralarm;
    new_action.sa_mask=sigset;
    new_action.sa_flags=0;
    sigaction(SIGALRM,&new_action,&old_action);
    gotsenderror=0;
    alarm(5); /* Alarm too!! */
    while (remain>0)
    {
#ifdef USE_WINSOCK
	if ((glen=send(thesock,this_msg,remain,0))<=0)
#else
	if ((glen=write(thesock,this_msg,remain))<=0)
#endif
	{
	  if (gotsenderror)
	    {
	      close_connection();
	      return remain;
	    }
	    if (glen<0) /* Error has occured. */
	    {
	      if (errno!=EINTR) /* If we got EINTR, we just try again. */
		{
		  close_connection();
		  return remain;
		}
	      glen=0;
	    }
	}
	remain-=glen;
	this_msg+=glen;
    }
    alarm(0); /* Turn off alarm. */
    sigaction(SIGALRM,&old_action,NULL);
    return 0;
}

static int gotconnecterror=0;

static void connecterroralarm(int signum)
{
  gotconnecterror=1;
}

static SOCKET open_connection(myuint32_t hostaddress,int myport, int maxtime)
{
  struct sockaddr_in cli_addr;
  SOCKET mysock=-1;
  struct sigaction old_action;
  struct sigaction new_action;
  sigset_t sigset;
  myuint32_t addr=htonl(hostaddress);
  sigemptyset(&sigset);
  new_action.sa_handler=connecterroralarm;
  new_action.sa_mask=sigset;
  new_action.sa_flags=0;
  sigaction(SIGALRM,&new_action,&old_action);
  gotconnecterror=0;
  alarm(maxtime); /* Alarm too!! */
  if ((mysock=socket(PF_INET,SOCK_STREAM,0))<0)
    goto end;
  cli_addr.sin_family=PF_INET;
  cli_addr.sin_port=htons(myport);
  memcpy(&cli_addr.sin_addr,&addr,sizeof(addr));
  if ((connect(mysock,(struct sockaddr*) &cli_addr,sizeof(cli_addr))<0)
      || (gotconnecterror))
    {
      closesocket(mysock);
      mysock=-1;
    }
 end:
  alarm(0); /* Turn off alarm. */
  sigaction(SIGALRM,&old_action,NULL);
  return mysock;
}


#endif /* USE_BSDSOCK */

void ymolsock_init()
{
#ifdef USE_BSDSOCK
  init_failure=0;
  signal(13,SIG_IGN); /* Ignore sigpipe! */
#ifdef USE_WINSOCK
  WSADATA wsaData;
  if (WSAStartup(MAKEWORD(1,1),&wsaData))
    {
      printf("No suitable winsock implementation.\n");
      init_failure=1;
    }
  printf("Found WINSOCK %d.%d\n",LOBYTE(wsaData.wVersion),HIBYTE(wsaData.wVersion));
  if ((LOBYTE(wsaData.wVersion)!=1) || (HIBYTE(wsaData.wVersion)!=1))
    {
      printf("No suitable winsock implementation.\n");
      WSACleanup();
      init_failure=1;
    }   
#endif /* WINSOCK */
#endif /* USE_BSDSOCK */
}


void ymolsock_deinit()
{
#ifdef USE_BSDSOCK
  if (!init_failure)
    {
      close_connection();
#ifdef USE_WINSOCK
      WSACleanup();
#endif /* WINSOCK */
    }
#endif /* BSDSOCK */
}

int ymolsock_check_ymol()
{
#ifdef USE_BSDSOCK
  if (!init_failure)
    {
      if (has_connection)
	return 1;
      else
	{
	  char buf[1000];
	  FILE *inf;
	  fflush(stdout);
	  inf=fopen("ymol_sockimport","r");
	  if (inf)
	    {
	      if (fgets(buf,1000,inf)==buf)
		{
		  if (strlen(buf)==17)
		    {
		      strcpy(magic_cookie,buf);
		      if (fgets(buf,1000,inf)==buf)
			{
			  strncpy(ymol_hostname,buf,MYMAXHOSTNAMELEN);
			  if (!strlen(ymol_hostname))
			    {
			      close_connection();
			      return 0;
			    }
			  ymol_hostname[strlen(ymol_hostname)-1]=0;
			  if (fgets(buf,1000,inf)==buf)
			    {
			      fclose(inf);
			      if (1==sscanf(buf,"%d",&PORT))
				{
				  /* Try to connect */
				  ymol_host=gethostbyname(ymol_hostname);
				  if (!ymol_host)
				    {
				      perror("gethostbyname");
				      {
					close_connection();
					return 0;
				      }
				    }
				  if (ymol_host->h_length!=4)
				    {
				      close_connection();
				      return 0;
				    }
				  ymol_address=ntohl(*(myuint32_t*)ymol_host->h_addr);
				  connsock=open_connection(ymol_address,PORT,1);
				  if (connsock<0)
				    {
				      close_connection();
				      remove("ymol_sockimport");
				      return 0;
				    }
				  has_connection=1;
				  if (write_any_sock(magic_cookie,17,connsock)!=0)
				    {
				      close_connection();
				      remove("ymol_sockimport");
				      return 0;
				    }
				  return 1;
				}
			    }
			}
		    }
		}
	      fclose(inf);
	    }
	}
    }
#endif
  return 0;
}

int ymolsock_write_natoms(int natoms)
{
#ifdef USE_BSDSOCK
  if (has_connection)
    {
      char buf[100];
      int l;
      sprintf(buf,"%d\n",natoms);
      l=strlen(buf);
      if (write_any_sock(buf,l,connsock)!=0)
	{
	  close_connection();
	  return 0;
	}
      return 1;
    }
#endif 
  return 0;
}

int ymolsock_write_natoms_nbonds(int natoms, int nbonds)
{
#ifdef USE_BSDSOCK
  if (has_connection)
    {
      char buf[100];
      int l;
      sprintf(buf,"-1\n");
      l=strlen(buf);
      if (write_any_sock(buf,l,connsock)!=0)
	{
	  close_connection();
	  return 0;
	}
      sprintf(buf,"%d %d\n",natoms,nbonds);
      l=strlen(buf);
      if (write_any_sock(buf,l,connsock)!=0)
	{
	  close_connection();
	  return 0;
	}
      return 1;
    }
#endif 
  return 0;
}


int ymolsock_write_atom(int id, int Z,double x, double y, double z)
{
#ifdef USE_BSDSOCK
  if (has_connection)
    {
      char buf[1000];
      int l;
      sprintf(buf,"%d %d %g %g %g\n",id,Z,x,y,z);
      l=strlen(buf);
      if (write_any_sock(buf,l,connsock)!=0)
	{
	  close_connection();
	  return 0;
	}
      return 1;
    }
#endif 
  return 0;
}

int ymolsock_write_bond(int i1,int i2, double r, double g, double b, double rsv, int n, double rad)
{
#ifdef USE_BSDSOCK
  if (has_connection)
    {
      char buf[1000];
      int l;
      sprintf(buf,"%d %d %g %g %g %g %d %g\n",i1,i2,r,g,b,rsv,n,rad);
      l=strlen(buf);
      if (write_any_sock(buf,l,connsock)!=0)
	{
	  close_connection();
	  return 0;
	}
      return 1;
    }
#endif 
  return 0;
}

