/* Calculate polygons from polygon rules. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "worldmemory.h"
#include "polygonrules.h"

#include <X11/Xlib.h>
#include "Xco.h"
#include "atomlabel.h"
#include "periodic_table.h"
#include "defaults.h"
#include "ccinterface.h"
#include "register_update.h"

#define MAXCENTRAL 100
#define MAXLIGANDS 100
#define MAXRULES 100
#define MAXNAME 64

struct polygonrule
{
  char name[MAXNAME];
  int Zcentral[MAXCENTRAL];
  int ncentral;
  int Zligand[MAXLIGANDS];
  int nligand;
  double r_central_ligand;
  double r_ligand_ligand;
  double r_min_distance;
  double rsv;
  double scalerad;
  double centralscalerad;
  double otherscalerad;
  int n;
  double trans;
  int ir, ig, ib;
};

struct triangle
{
  struct polygonrule *pr;
  int iligand0, iligand1, iligand2;
  double x0,y0,z0,x1,y1,z1,x2,y2,z2;
  struct triangle *next;
};

static int *neighbour_list=NULL;
static int *neighbour_head=NULL;
static int *neighbour_num=NULL;

static int numrules=0;
static struct polygonrule polygonrule[MAXRULES];
static struct triangle *triangle=NULL;

int get_number_of_polygon_rules()
{
  return numrules;
}

extern void pairgt(int *pair, int *pair1, int *pair2);

static int check_ligand_rule(struct polygonrule *pr,int Z)
{
  int i;
  int iz=0;
  for (i=0; i<pr->nligand; i++)
    if (Z==pr->Zligand[i])
      {
	iz=1;
	break;
      }
  return iz;
}

static int check_central_rule(struct polygonrule *pr,int ZC)
{
  int i;
  int izc=0;
  for (i=0; i<pr->ncentral; i++)
    if (ZC==pr->Zcentral[i])
      {
	izc=1;
	break;
      }
  return izc;
}

static int dcheck(double x0,double y0,double z0,double x1,double y1,double z1,double r)
{
  double r2=r*r;
  double dx=x1-x0;
  double dy=y1-y0;
  double dz=z1-z0;
  if (dx*dx+dy*dy+dz*dz<r2)
    return 1;
  else
    return 0;
}

static void crossp(double x0, double y0, double z0, 
		   double x1, double y1, double z1, 
		   double *nx, double *ny, double *nz)
{
  *nx=y0*z1-y1*z0;
  *ny=z0*x1-z1*x0;
  *nz=x0*y1-x1*y0;
}

static int check_plane_distance(double x0,double y0,double z0,double x1,double y1,double z1,double x2, double y2, double z2, double xc, double yc, double zc, double rmin)
{
  double rplanepoint;
  double a,b,c,d;
  /* Form the equation of a plane ax+by+cz+d=0 */
  crossp(x1-x0,y1-y0,z1-z0,x2-x0,y2-y0,z2-z0,&a,&b,&c);
  d=-a*x0-b*y0-c*z0;
  /* What is the distance from the central atom point (xc,yc,zc) to the plane? */
  rplanepoint=fabs(a*xc+b*yc+c*zc+d)/sqrt(a*a+b*b+c*c);
  if (rplanepoint<rmin)
    return 0;
  else
    return 1;
}

#if 0
static double cosdihed(double x0, double y0, double z0,
		       double x1, double y1, double z1,
		       double x2, double y2, double z2,
		       double x3, double y3, double z3)
{
  double d1x=x1-x0;
  double d1y=y1-y0;
  double d1z=z1-z0;
  double d2x=x2-x1;
  double d2y=y2-y1;
  double d2z=z2-z1;
  double d3x=x3-x2;
  double d3y=y3-y2;
  double d3z=z3-z2;

  /* Dot products. */
  double c11=d1x*d1x+d1y*d1y+d1z*d1z;
  double c12=d1x*d2x+d1y*d2y+d1z*d2z;
  double c13=d1x*d3x+d1y*d3y+d1z*d3z;
  double c22=d2x*d2x+d2y*d2y+d2z*d2z;
  double c23=d2x*d3x+d2y*d3y+d2z*d3z;
  double c33=d3x*d3x+d3y*d3y+d3z*d3z;

  double cA=c13*c22-c12*c23;
  double cB1=c11*c22-c12*c12;
  double cB2=c22*c33-c23*c23;
  double cD=sqrt(cB1*cB2);
  double c=cA/cD;
  return c;
}
#endif

static int check_previous_triangles(int iligand0, int iligand1, int iligand2, int frame)
{
  struct triangle *t;
  for (t=triangle; t; t=t->next)
    {
      int isame[3], jsame[3];
      int numsame=0;
      /* Does this triangle share two atoms with the new triangle? */
      if ((t->iligand0==iligand0) ||
	  (t->iligand1==iligand0) ||
	  (t->iligand2==iligand0))
	{
	  isame[numsame]=iligand0;
	  if (t->iligand0==iligand0)
	    jsame[numsame]=t->iligand0;
	  else if (t->iligand1==iligand0)
	    jsame[numsame]=t->iligand1;
	  else if (t->iligand2==iligand0)
	    jsame[numsame]=t->iligand2;
	  numsame++;
	}
      if ((t->iligand0==iligand1) ||
	  (t->iligand1==iligand1) ||
	  (t->iligand2==iligand1))
	{
	  isame[numsame]=iligand1;
	  if (t->iligand0==iligand1)
	    jsame[numsame]=t->iligand0;
	  else if (t->iligand1==iligand1)
	    jsame[numsame]=t->iligand1;
	  else if (t->iligand2==iligand1)
	    jsame[numsame]=t->iligand2;
	  numsame++;
	}
      if ((t->iligand0==iligand2) ||
	  (t->iligand1==iligand2) ||
	  (t->iligand2==iligand2))
	{
	  isame[numsame]=iligand2;
	  if (t->iligand0==iligand2)
	    jsame[numsame]=t->iligand0;
	  else if (t->iligand1==iligand2)
	    jsame[numsame]=t->iligand1;
	  else if (t->iligand2==iligand2)
	    jsame[numsame]=t->iligand2;
	  numsame++;
	}
      if (numsame==2)
	{
	  double a0,b0,c0;
	  double a1,b1,c1;
	  double x0,y0,z0,x1,y1,z1,x2,y2,z2,x3,y3,z3;
	  double cost;
	  /* Which atoms are not the same? */
	  int inotsame, jnotsame;
	  int imask[3], jmask[3];
	  int i;
	  for (i=0; i<3; i++)
	    imask[i]=1;
	  for (i=0; i<3; i++)
	    jmask[i]=1;
	  if (isame[0]==iligand0)
	    imask[0]=0;
	  if (isame[1]==iligand0)
	    imask[0]=0;
	  if (isame[0]==iligand1)
	    imask[1]=0;
	  if (isame[1]==iligand1)
	    imask[1]=0;
	  if (isame[0]==iligand2)
	    imask[2]=0;
	  if (isame[1]==iligand2)
	    imask[2]=0;
	  if (jsame[0]==t->iligand0)
	    jmask[0]=0;
	  if (jsame[1]==t->iligand0)
	    jmask[0]=0;
	  if (jsame[0]==t->iligand1)
	    jmask[1]=0;
	  if (jsame[1]==t->iligand1)
	    jmask[1]=0;
	  if (jsame[0]==t->iligand2)
	    jmask[2]=0;
	  if (jsame[1]==t->iligand2)
	    jmask[2]=0;
	  if (imask[0])
	    inotsame=iligand0;
	  if (imask[1])
	    inotsame=iligand1;
	  if (imask[2])
	    inotsame=iligand2;
	  if (jmask[0])
	    jnotsame=t->iligand0;
	  if (jmask[1])
	    jnotsame=t->iligand1;
	  if (jmask[2])
	    jnotsame=t->iligand2;
	  get_atomxyz(&frame,&isame[0],&x0,&y0,&z0);
	  get_atomxyz(&frame,&isame[1],&x1,&y1,&z1);
	  get_atomxyz(&frame,&inotsame,&x2,&y2,&z2);
	  get_atomxyz(&frame,&jnotsame,&x3,&y3,&z3);

#if 0	  
	  cost=cosdihed(x2,y2,z2,x0,y0,z0,x1,y1,z1,x3,y3,z3);
	  if (cost>0)
	    return 0;
#endif
#if 1
	  /* Get the normals of the planes. */
	  crossp(x1-x0,y1-y0,z1-z0,x2-x0,y2-y0,z2-z0,&a0,&b0,&c0);
	  crossp(x1-x0,y1-y0,z1-z0,x3-x0,y3-y0,z3-z0,&a1,&b1,&c1);
	  /* Cosine of the dihedral angle */
	  cost=(a0*a1+b0*b1+c0*c1)/(sqrt(a0*a0+b0*b0+c0*c0)*sqrt(a1*a1+b1*b1+c1*c1));
	  if (cost>0.75)
	    return 0;
#endif
	}
    }
  return 1;
}

static void check_add_triangle(struct polygonrule *pr, int iligand0, int iligand1, int iligand2, int icentral, int frame)
{
  struct triangle *t;
  int add_it=1;
  double x0,y0,z0,x1,y1,z1,x2,y2,z2;
  double xc,yc,zc;
  for (t=triangle; t; t=t->next)
    {
      /* Never add a triangle twice */
      int has0=0, has1=0, has2=0;
      if ((t->iligand0==iligand0) ||
	  (t->iligand1==iligand0) ||
	  (t->iligand2==iligand0))
	has0=1;
      if ((t->iligand0==iligand1) ||
	  (t->iligand1==iligand1) ||
	  (t->iligand2==iligand1))
	has1=1;
      if ((t->iligand0==iligand2) ||
	  (t->iligand1==iligand2) ||
	  (t->iligand2==iligand2))
	has2=1;
      if (has0+has1+has2==3)
	{
	  add_it=0;
	  break;
	}
    }
  if (add_it)
    {
      get_atomxyz(&frame,&iligand0,&x0,&y0,&z0);
      get_atomxyz(&frame,&iligand1,&x1,&y1,&z1);
      get_atomxyz(&frame,&iligand2,&x2,&y2,&z2);
      get_atomxyz(&frame,&icentral,&xc,&yc,&zc);
      /* Never add triangles for atoms that are (close to) the same
	 plane of the central atom. */
      if (check_plane_distance(x0,y0,z0,x1,y1,z1,x2,y2,z2,xc,yc,zc,pr->r_min_distance))
	{
	  /* Never add triangles which share two atoms with a previous
	     triangle and the torsion angle is less than 90
	     degrees. Avoids double triangles when four atoms are
	     close. */
	  if (check_previous_triangles(iligand0,iligand1,iligand2,frame))
	    {
	      struct triangle *new;
	      double testrad;
	      new=malloc(sizeof *new);
	      new->next=triangle;
	      triangle=new;
	      new->iligand0=iligand0;
	      new->iligand1=iligand1;
	      new->iligand2=iligand2;
	      new->x0=x0;
	      new->y0=y0;
	      new->z0=z0;
	      new->x1=x1;
	      new->y1=y1;
	      new->z1=z1;
	      new->x2=x2;
	      new->y2=y2;
	      new->z2=z2;
	      new->pr=pr;
	      /* Modify the scaling radii for the ligand atoms. */
	      /* And the central atom */
	      get_atomscalerad(&frame,&iligand0,&testrad);
	      if (fabs(testrad-1.0)<1e-7)
		set_atomscalerad(&frame,&iligand0,&pr->scalerad);
	      get_atomscalerad(&frame,&iligand1,&testrad);
	      if (fabs(testrad-1.0)<1e-7)
		set_atomscalerad(&frame,&iligand1,&pr->scalerad);
	      get_atomscalerad(&frame,&iligand2,&testrad);
	      if (fabs(testrad-1.0)<1e-7)
		set_atomscalerad(&frame,&iligand2,&pr->scalerad);
	      get_atomscalerad(&frame,&icentral,&testrad);
	      if (fabs(testrad-1.0)<1e-7)
		set_atomscalerad(&frame,&icentral,&pr->centralscalerad);
	    }
	}
    }
}

static void clear_triangles()
{
  struct triangle *t, *tmp;
  t=triangle;
  while (t)
    {
      tmp=t;
      t=t->next;
      free(tmp);
    }
  triangle=NULL;
}

static void add_triangles_to_frame(int frame)
{
  int ntriangles=0;
  struct triangle *t;
  const double trans_limit=1e-3;
  /* Count number of triangles. */
  for (t=triangle; t; t=t->next)
    if (t->pr->trans>trans_limit)
      ntriangles++;
  reallocate_triangles(frame,ntriangles*3);
#if 0
  printf("Polygon rules found %d triangles.\n",ntriangles);
#endif
  ntriangles=0;
  for (t=triangle; t; t=t->next)
    {
      /* If this is completely transparent, actually do not add the triangle, could still be useful to only use the polygon rules, but
	 not adding the actual polygon! */
      if (t->pr->trans>trans_limit)
	{
	  double nx,ny,nz,inl;
	  /* Compute the normal of the triangle using the cross product. */
	  double p11=t->x1-t->x0;
	  double p12=t->y1-t->y0;
	  double p13=t->z1-t->z0;
	  double p21=t->x2-t->x0;
	  double p22=t->y2-t->y0;
	  double p23=t->z2-t->z0;
	  crossp(p11,p12,p13,p21,p22,p23,&nx,&ny,&nz);
	  /* Normalize */
	  inl=1./sqrt(nx*nx+ny*ny+nz*nz);
	  nx*=inl;
	  ny*=inl;
	  nz*=inl;
	  set_frame_triangle(frame,ntriangles++,t->x0,t->y0,t->z0,nx,ny,nz,t->pr->rsv,t->pr->trans,t->pr->ir,t->pr->ig,t->pr->ib,t->pr->n,1);
	  set_frame_triangle(frame,ntriangles++,t->x1,t->y1,t->z1,nx,ny,nz,t->pr->rsv,t->pr->trans,t->pr->ir,t->pr->ig,t->pr->ib,t->pr->n,1);
	  set_frame_triangle(frame,ntriangles++,t->x2,t->y2,t->z2,nx,ny,nz,t->pr->rsv,t->pr->trans,t->pr->ir,t->pr->ig,t->pr->ib,t->pr->n,1);
	}
    }
}

static void set_other_atom_scaling(int frame,struct polygonrule *pr)
{
  int numatoms;
  int i;
  /* Scaling for all other atoms. */
  get_atoms(&frame,&numatoms);
  for (i=0; i<numatoms; i++)
    {
      double testrad;
      get_atomscalerad(&frame,&i,&testrad);
      if (fabs(testrad-1.0)<1e-7)
	set_atomscalerad(&frame,&i,&pr->otherscalerad);
    }
}

static void create_neighbour_list(int frame, double rmax)
{
  int numatoms;
  int npairs;
  int i, istart;
  get_atoms(&frame,&numatoms);
  compute_pairs_frame(frame,rmax,&npairs);
  neighbour_list=malloc(npairs*2*sizeof *neighbour_list);
  neighbour_num=malloc(numatoms*sizeof *neighbour_num);
  neighbour_head=malloc(numatoms*sizeof *neighbour_head);
  memset(neighbour_num,0,numatoms*sizeof *neighbour_num);
  /* First count the number of neighbours for each atom. */
  for (i=0; i<npairs; i++)
    {
      int i0,i1;
      pairgt(&i,&i0,&i1);
      if (i0!=i1)
	{
	  neighbour_num[i0]++;
	  neighbour_num[i1]++;
	}
    }
  /* Compute head position in list. */
  istart=0;
  for (i=0; i<numatoms; i++)
    {
      neighbour_head[i]=istart;
      istart+=neighbour_num[i];
    }
  /* Create the list. */
  memset(neighbour_num,0,numatoms*sizeof *neighbour_num);
  for (i=0; i<npairs; i++)
    {
      int i0,i1;
      pairgt(&i,&i0,&i1);
      if (i0!=i1)
	{
	  neighbour_list[neighbour_head[i0]+neighbour_num[i0]]=i1;
	  neighbour_list[neighbour_head[i1]+neighbour_num[i1]]=i0;
	  neighbour_num[i0]++;
	  neighbour_num[i1]++;
	}
    }
}

void apply_polygonrules(int frame)
{
  if (numrules)
    {
      double rmax=0.;
      int i;
      int numatoms;
      clear_triangles();
      get_atoms(&frame,&numatoms);
      /* Default scaling is 1.0. */
      for (i=0; i<numatoms; i++)
	{
	  double one=1.0;
	  set_atomscalerad(&frame,&i,&one);
	}
      for (i=0; i<numrules; i++)
	{
	  if (polygonrule[i].r_central_ligand>rmax)
	    rmax=polygonrule[i].r_central_ligand;
	  if (polygonrule[i].r_ligand_ligand>rmax)
	    rmax=polygonrule[i].r_ligand_ligand;
	}
      create_neighbour_list(frame,rmax);
      for (i=0; i<numrules; i++)
	{
	  int icentral;
	  struct polygonrule *pr=polygonrule+i;
	  /* Find all central atoms for this rule. */
	  for (icentral=0; icentral<numatoms; icentral++)
	    {
	      int icentral_Z;
	      get_atomnr(&frame,&icentral,&icentral_Z);
	      if (check_central_rule(pr,icentral_Z))
		{
		  /* Now find all triplets of neighbours for this atom. */
		  double xc,yc,zc;
		  int i0,i1,i2;
		  get_atomxyz(&frame,&icentral,&xc,&yc,&zc);
		  for (i0=0; i0<neighbour_num[icentral]; i0++)
		    {
		      for (i1=i0+1; i1<neighbour_num[icentral]; i1++)
			{
			  for (i2=i1+1; i2<neighbour_num[icentral]; i2++)
			    {
			      int iligand0,iligand1,iligand2;
			      int iligand0_Z,iligand1_Z,iligand2_Z;
			      double x0,y0,z0, x1,y1,z1, x2,y2,z2;
			      iligand0=neighbour_list[neighbour_head[icentral]+i0];
			      iligand1=neighbour_list[neighbour_head[icentral]+i1];
			      iligand2=neighbour_list[neighbour_head[icentral]+i2];
			      get_atomnr(&frame,&iligand0,&iligand0_Z);
			      get_atomnr(&frame,&iligand1,&iligand1_Z);
			      get_atomnr(&frame,&iligand2,&iligand2_Z);
			      get_atomxyz(&frame,&iligand0,&x0,&y0,&z0);
			      get_atomxyz(&frame,&iligand1,&x1,&y1,&z1);
			      get_atomxyz(&frame,&iligand2,&x2,&y2,&z2);
			      /* The types of atoms? */
			      if ((check_ligand_rule(pr,iligand0_Z)) &&
				  (check_ligand_rule(pr,iligand1_Z)) &&
				  (check_ligand_rule(pr,iligand2_Z)))
				{
				  /* Distances to the central atom ok? */
				  if ((dcheck(xc,yc,zc,x0,y0,z0,pr->r_central_ligand)) &&
				      (dcheck(xc,yc,zc,x1,y1,z1,pr->r_central_ligand)) &&
				      (dcheck(xc,yc,zc,x2,y2,z2,pr->r_central_ligand)))
				    {
				      /* Ligand-ligand distances? */
				      if ((dcheck(x0,y0,z0,x1,y1,z1,pr->r_ligand_ligand)) &&
					  (dcheck(x0,y0,z0,x2,y2,z2,pr->r_ligand_ligand)) &&
					  (dcheck(x1,y1,z1,x2,y2,z2,pr->r_ligand_ligand)))
					{
					  check_add_triangle(pr,iligand0,iligand1,iligand2,icentral,frame);
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	  set_other_atom_scaling(frame,pr);
	}
      free(neighbour_list);
      free(neighbour_head);
      free(neighbour_num);
      add_triangles_to_frame(frame);
    }
#if 1
  else
    {
      int numatoms;
      double one=1.;
      int i;
      reallocate_triangles(frame,0);
      get_atoms(&frame,&numatoms);
      for (i=0; i<numatoms; i++)
	set_atomscalerad(&frame,&i,&one);
    }
#endif
}

/* F77 interface routines. */
void maprf_(int *frame)
{
  apply_polygonrules(*frame);
}

void gnpr_(int *n)
{
  *n=numrules;
}

void snpr_(int *n)
{
  numrules=*n;
}

void gprnam_(int *rule, char *n, int len)
{
  int i;
  int mlen=strlen(polygonrule[*rule].name);
  strcpy(n,polygonrule[*rule].name);
  for (i=mlen; i<len; i++)
    n[i]=' ';
}

void sprnam_(int *rule, char *n, int len)
{
  int i;
  strncpy(polygonrule[*rule].name,n,len);
  for (i=len-2; i>0; i--)
    if (n[i]!=' ')
      break;
  n[i+1]=0;
}

void gprnc_(int *rule, int *n)
{
  *n=polygonrule[*rule].ncentral;
}

void sprnc_(int *rule, int *n)
{
  polygonrule[*rule].ncentral=*n;
}

void gprnl_(int *rule, int *n)
{
  *n=polygonrule[*rule].nligand;
}

void sprnl_(int *rule, int *n)
{
  polygonrule[*rule].nligand=*n;
}

void gprcz_(int *rule, int *n, int *z)
{
  *z=polygonrule[*rule].Zcentral[*n];
}

void sprcz_(int *rule, int *n, int *z)
{
  polygonrule[*rule].Zcentral[*n]=*z;
}

void gprlz_(int *rule, int *n, int *z)
{
  *z=polygonrule[*rule].Zligand[*n];
}

void sprlz_(int *rule, int *n, int *z)
{
  polygonrule[*rule].Zligand[*n]=*z;
}

void gprp_(int *rule,
	  double *r_central_ligand,
	  double *r_ligand_ligand,
	  double *r_min_distance,
	  double *rsv,
	  double *scalerad,
	  double *centralscalerad,
	  double *otherscalerad,
	  int *n,
	  double *trans,
	  int *ir, int *ig, int *ib)
{
  *r_central_ligand=polygonrule[*rule].r_central_ligand;
  *r_ligand_ligand=polygonrule[*rule].r_ligand_ligand;
  *r_min_distance=polygonrule[*rule].r_min_distance;
  *rsv=polygonrule[*rule].rsv;
  *scalerad=polygonrule[*rule].scalerad;
  *centralscalerad=polygonrule[*rule].centralscalerad;
  *otherscalerad=polygonrule[*rule].otherscalerad;
  *n=polygonrule[*rule].n;
  *trans=polygonrule[*rule].trans;
  *ir=polygonrule[*rule].ir;
  *ig=polygonrule[*rule].ig;
  *ib=polygonrule[*rule].ib;
}

void sprp_(int *rule,
	  double *r_central_ligand,
	  double *r_ligand_ligand,
	  double *r_min_distance,
	  double *rsv,
	  double *scalerad,
	  double *centralscalerad,
	  double *otherscalerad,
	  int *n,
	  double *trans,
	  int *ir, int *ig, int *ib)
{
  polygonrule[*rule].r_central_ligand=*r_central_ligand;
  polygonrule[*rule].r_ligand_ligand=*r_ligand_ligand;
  polygonrule[*rule].r_min_distance=*r_min_distance;
  polygonrule[*rule].rsv=*rsv;
  polygonrule[*rule].scalerad=*scalerad;
  polygonrule[*rule].centralscalerad=*centralscalerad;
  polygonrule[*rule].otherscalerad=*otherscalerad;
  polygonrule[*rule].n=*n;
  polygonrule[*rule].trans=*trans;
  polygonrule[*rule].ir=*ir;
  polygonrule[*rule].ig=*ig;
  polygonrule[*rule].ib=*ib;
}

/* GUI */
#define WINDOW_WIDTH 330
#define WINDOW_HEIGHT 250

#define ADD_WINDOW_WIDTH 400
#define ADD_WINDOW_HEIGHT 580

static int add_rule_open=0;
static XcoObject add_window,namedialog,add_decorbox,
  centraldialog,liganddialog,centralliganddialog,ligandliganddialog,rmindistancedialog,colorhole;
static XcoObject transdialog,phongndialog,rsvdialog,scaleraddialog,centralscaleraddialog,otherscaleraddialog;

static int open_bcolor=0;
static XcoObject colsel;
static int current_item;
static int delete_if_cancel;
static int use_default_color;

static int polygon_rules_open=0;

static XcoObject top_window,decorbox;
static XcoObject list;

static char **list_items;
static int list_no;

static int browse_pixel; 
static Pixmap colorpixmap;
static int colorpixmap_exist=0;

int virgin=1;

static void make_list()
{
  list_no=numrules;
  if (list_no>0)
    {
      int i;
      list_items=malloc(sizeof( char*)*(list_no));
      for (i=1; i<=list_no;i++)
	{
	  list_items[i-1]=malloc(sizeof( char)*(MAXNAME));
	  strcpy(list_items[i-1],polygonrule[i-1].name);
	}
    }
  else
    list_items=NULL;
}

static void delete_list()
{
  if (list_no>0)
    {
      int i;
      for (i=1; i<=list_no; i++)
	{
	  free( list_items[i-1]);
	}
      free( list_items);
      list_no=0;
      list_items=NULL;
    }
}

static void close_add_rule_window()
{
  XcoDeleteObject(add_window);
  add_rule_open=0;
  if (open_bcolor)
    {
      XcoDeleteObject(colsel);
      open_bcolor=0;
    }
}

/* Create central/ligand string */
static void centralligandstring(int item,int central,char *s)
{
  int sptr=0;
  int istr=0;
  if (central)
    istr=polygonrule[item].ncentral;
  else
    istr=polygonrule[item].nligand;
  if (istr==0)
    {
      sprintf(s,"None");
    }
  else
    {
      int i;
      for (i=1; i<=istr; i++)
	{
	  int inum;
	  int lblp=0;
	  char lbl[4];
	  if (central)
	    inum=polygonrule[item].Zcentral[i-1];
	  else
	    inum=polygonrule[item].Zligand[i-1];

	  get_atomlabel(inum,lbl);

	  while (lbl[lblp]!='\0')
	    s[sptr++]=lbl[lblp++];
	  if (i!=istr)
	    s[sptr++]=' ';
	}
      s[sptr++]='\0';
    }
}      

/* Analyze and set central/ligand string */
static void set_centralligand(int item,int central,char *s)
{
  int sptr=0;
  int istr=0;
  int ilen=strlen(s);
  int anum=1;

  char aname[4];
  int wptr=0;
  while (sptr<ilen)
    {
      char lbl[4];
      /* remove spaces */
      while ((sptr<ilen) && (s[sptr]==' '))
	sptr++;
      
      while ((sptr<ilen) && (wptr<3) && (s[sptr]!=' '))
        {
	  aname[wptr++]=s[sptr++];
	}
      aname[wptr]='\0';
      /*  printf("Found: '%s'\n",aname); */


      get_atomlabel(anum,lbl);
      while ((anum<N_ATOMS) && (strcmp(lbl,aname)!=0))
	{
	  anum++;
	  get_atomlabel(anum,lbl);
	}
      if (strcmp(lbl,aname)==0)
	{
	  istr++;
	  if (central)
	    polygonrule[item].Zcentral[istr-1]=anum;
	  else
	    polygonrule[item].Zligand[istr-1]=anum;
	}
      else
	{
	  /* printf("Error\n"); */
	}
      wptr=0;
    }
  if (central)
    polygonrule[item].ncentral=istr;
  else
    polygonrule[item].nligand=istr;
}      


static void polygon_delete_window(XcoObject id,XEvent event)
{
  if (XcoDeleteWindow(id,event))
    {
      if (!add_rule_open)
	{
	  XcoDeleteObject(top_window);
	  delete_list();
	  polygon_rules_open=0;
	}
    }
}

static void polygon_apply(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      mapr_();
      wupd_();
    }
}

static void polygon_apply_frame(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int frame,nframes;
      gframe_(&frame,&nframes);
      maprf_(&frame);
      wupd_();
    }
}

static void delete_one_rule(int item)
{
  int i;
  for (i=item; i<numrules-1; i++)
    polygonrule[i]=polygonrule[i+1];
  numrules--;
}

static void polygon_delete(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int status=XcoGetListStatus(list);
      if (status!=-1)
	{
	  if (!add_rule_open)
	    {
	      delete_one_rule(status);
	      delete_list();
	      make_list();
	      XcoSetListData(list,list_items,list_no);
	    }
	}
    }
}

static void polygon_up(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int status=XcoGetListStatus(list);
      if (status!=-1)
	{
	  if (!add_rule_open)
	    {
	      if (status>=1)
		{
		  struct polygonrule tmp=polygonrule[status-1];
		  polygonrule[status-1]=polygonrule[status];
		  polygonrule[status]=tmp;
		  delete_list();
		  make_list();
		  XcoSetListData(list,list_items,list_no);
		}
	    }
	}
    }
}

static void polygon_down(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int status=XcoGetListStatus(list);
      if (status!=-1)
	{
	  if (!add_rule_open)
	    {
	      if (status<numrules-1)
		{
		  struct polygonrule tmp=polygonrule[status+1];
		  polygonrule[status+1]=polygonrule[status];
		  polygonrule[status]=tmp;
		  delete_list();
		  make_list();
		  XcoSetListData(list,list_items,list_no);
		}
	    }
	}
    }
}


static void add_delete_window(XcoObject id,XEvent event)
{
  if (XcoDeleteWindow(id,event))
    {
      if (delete_if_cancel)
	delete_one_rule(current_item);
      delete_list();
      make_list();
      XcoSetListData(list,list_items,list_no);
      close_add_rule_window();
    }
}

static void cbrowse(int *alist)
{
  int i;
  int n=alist[0];
  polygonrule[current_item].ncentral=n;
  
  for (i=1; i<=n; i++)
    {
      polygonrule[current_item].Zcentral[i-1]=alist[i];
    }
  if (add_rule_open)
    {
      char dnr[300];
      centralligandstring(current_item,1,dnr);
      XcoSetDialogValue(centraldialog,dnr);
    }
}

static void lbrowse(int *alist)
{
  int i;
  int n=alist[0];

  polygonrule[current_item].nligand=n;
  for (i=1; i<=n; i++)
    {
      polygonrule[current_item].Zligand[i-1]=alist[i];
    }
  if (add_rule_open)
    {
      char dnr[300];
      centralligandstring(current_item,0,dnr);
      XcoSetDialogValue(liganddialog,dnr);
    }
}

static void browse_central(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int istr;
      int *ilist;
      set_centralligand(current_item,1,XcoGetDialogValue(centraldialog));

      istr=polygonrule[current_item].ncentral;

      if (istr==0)
	{
	  ilist=NULL;
	}
      else
	{
	  int i;
	  ilist=malloc(sizeof( int)*(istr));
	  for (i=1; i<=istr; i++)
	    ilist[i-1]=polygonrule[current_item].Zcentral[i-1];
	}

      open_periodic_table("Select central atoms",cbrowse,100,ilist,istr);
      if (ilist!=NULL)
	free( ilist);
    }
}

static void browse_ligand(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int istr;
      int *ilist;
      set_centralligand(current_item,0,XcoGetDialogValue(liganddialog));
      istr=polygonrule[current_item].nligand;

      if (istr==0)
	{
	  ilist=NULL;
	}
      else
	{
	  int i;
	  ilist=malloc(sizeof( int)*(istr));
	  for (i=1; i<=istr; i++)
	    ilist[i-1]=polygonrule[current_item].Zligand[i-1];
	}
      
      open_periodic_table("Select ligand atoms",lbrowse,100,ilist,istr);
      if (ilist!=NULL)
	free( ilist);
    }
}

static void set_colorpixmap(unsigned int pixel)
{
  int i;
  unsigned int image[16];
  browse_pixel=pixel;
  if (colorpixmap_exist)
    XFreePixmap(XcoGetDisplay(),colorpixmap);

  for (i=0; i<16; i++)
    image[i]=pixel;
  colorpixmap=XcoCreatePixmapFromImage(colorhole,image,4,4);
  XcoSetBackgroundPixmap(colorhole,colorpixmap,True);
}

static void use_color(int r,int g,int b,int ok)
{
  if (add_rule_open)
    if (ok)
      {
	set_colorpixmap(PIXEL(r,g,b));
      }
}

static void dcolsel(XcoObject dummy,XEvent event)
{
  if (event.type==DestroyNotify)
    {
      printf("The color selector object was destroyed\n");
      open_bcolor=0;
    }
}

static void browse_colors(XcoObject dummy,XEvent event)
{
  if (add_rule_open)
    {
      if (event.type==ButtonPress)
	{
	  if (!open_bcolor)
	    {
	      colsel=XcoColorselector(R_OF_PIXEL(browse_pixel),
				      G_OF_PIXEL(browse_pixel),
				      B_OF_PIXEL(browse_pixel),
				      use_color);
	      XcoAddCallback(colsel,dcolsel);
	      open_bcolor=1;
	    }
	}
    }
}

static void okbutton_callback(XcoObject dummy, XEvent event)
{
  if (event.type==ButtonPress)
    {
      strcpy(polygonrule[current_item].name,XcoGetDialogValue(namedialog));
      set_centralligand(current_item,1,XcoGetDialogValue(centraldialog));
      set_centralligand(current_item,0,XcoGetDialogValue(liganddialog));
      polygonrule[current_item].r_central_ligand=strtod(XcoGetDialogValue(centralliganddialog),NULL);
      polygonrule[current_item].r_ligand_ligand=strtod(XcoGetDialogValue(ligandliganddialog),NULL);
      polygonrule[current_item].r_min_distance=strtod(XcoGetDialogValue(rmindistancedialog),NULL);
      polygonrule[current_item].rsv=strtod(XcoGetDialogValue(rsvdialog),NULL);
      polygonrule[current_item].n=strtol(XcoGetDialogValue(phongndialog),NULL,10);
      polygonrule[current_item].trans=strtod(XcoGetDialogValue(transdialog),NULL);
      polygonrule[current_item].scalerad=strtod(XcoGetDialogValue(scaleraddialog),NULL);
      polygonrule[current_item].centralscalerad=strtod(XcoGetDialogValue(centralscaleraddialog),NULL);
      polygonrule[current_item].otherscalerad=strtod(XcoGetDialogValue(otherscaleraddialog),NULL);
      polygonrule[current_item].ir=R_OF_PIXEL(browse_pixel);
      polygonrule[current_item].ig=G_OF_PIXEL(browse_pixel);
      polygonrule[current_item].ib=B_OF_PIXEL(browse_pixel);
      
      delete_list();
      make_list();
      XcoSetListData(list,list_items,list_no);
      close_add_rule_window();
    }
}

static void cancelbutton_callback(XcoObject dummy, XEvent event)
{
  if (event.type==ButtonPress)
    {
      if (delete_if_cancel)
	delete_one_rule(current_item);
      delete_list();
      make_list();
      XcoSetListData(list,list_items,list_no);
      close_add_rule_window();
    }
}

static void modify_polygonrule(int item,char *wname,char *okbuttonname)
{
  if (!add_rule_open)
    {
      XcoObject hole,hole2,hole3,label,bcmd,colorbox,icol,cancelbutton,okbutton;
      int xpos,ypos;
      char name[64];
      char dnr[300];
      char acptr[300];
      char centralligands[5];
      char ligandligands[5];
      char rmindistances[5];
      current_item=item;
      use_default_color=1;
      add_window=XcoCreateNamedDialogParent(-1,0,0,ADD_WINDOW_WIDTH,ADD_WINDOW_HEIGHT,wname);

      XcoAddCallback(add_window,add_delete_window);

      add_decorbox=XcoCreateBox3D(add_window,5,5,ADD_WINDOW_WIDTH-10,ADD_WINDOW_HEIGHT-45,2,1);

      hole=XcoCreateHole(add_decorbox,5,5,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Name",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);

      strcpy(name,polygonrule[item].name);
      namedialog=XcoCreateDialog(hole2,0,ypos,add_window,name,63,ADD_WINDOW_WIDTH-20,0);

      hole=XcoCreateHole(add_decorbox,5,45,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Central atoms",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      hole3=XcoCreateHole(hole2,0,ypos,ADD_WINDOW_WIDTH-20,50-ypos);
      XcoSetResizeMethod(hole3,XcoUpResize);

      centralligandstring(item,1,dnr);
      centraldialog=XcoCreateDialog(hole3,0,0,add_window,dnr,300,ADD_WINDOW_WIDTH-100,0);
      bcmd=XcoCreateCommand(hole3,ADD_WINDOW_WIDTH-90,0,"Browse",0,0);
      XcoAddCallback(bcmd,browse_central);
      XcoSetResizeMethod(centraldialog,XcoUpLeftDownRight);
      XcoSetResizeMethod(bcmd,XcoUpRight);

      hole=XcoCreateHole(add_decorbox,5,85,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Ligandatoms",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      hole3=XcoCreateHole(hole2,0,ypos,ADD_WINDOW_WIDTH-20,50-ypos);
      XcoSetResizeMethod(hole3,XcoUpResize);

      centralligandstring(item,0,acptr);
      liganddialog=XcoCreateDialog(hole3,0,0,add_window,acptr,300,ADD_WINDOW_WIDTH-100,0);
      bcmd=XcoCreateCommand(hole3,ADD_WINDOW_WIDTH-90,0,"Browse",0,0);
      XcoAddCallback(bcmd,browse_ligand);
      XcoSetResizeMethod(liganddialog,XcoUpLeftDownRight);
      XcoSetResizeMethod(bcmd,XcoUpRight);

      hole=XcoCreateHole(add_decorbox,5,125,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Central-ligand distance criterion",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(centralligands,"%4.2f",polygonrule[item].r_central_ligand);
      centralliganddialog=XcoCreateDialog(hole2,0,ypos,add_window,centralligands,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,165,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Ligand-ligand distance criterion",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(ligandligands,"%4.2f",polygonrule[item].r_ligand_ligand);
      ligandliganddialog=XcoCreateDialog(hole2,0,ypos,add_window,ligandligands,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,205,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Minimum central atom - plane distance criterion",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].r_min_distance);
      rmindistancedialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,245,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Polygon color",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      hole3=XcoCreateHole(hole2,0,ypos,ADD_WINDOW_WIDTH-20,50-ypos);
      XcoSetResizeMethod(hole3,XcoUpResize);
      bcmd=XcoCreateCommand(hole3,60,0,"Browse",0,0);
      colorbox=XcoCreateBox3D(hole3,0,0,50,XcoGetObjectHeight(bcmd),0,1);
      colorhole=XcoCreateHole(colorbox,2,2,46,XcoGetObjectHeight(bcmd)-4);
      XcoAddCallback(bcmd,browse_colors);
      XcoSetResizeMethod(colorbox,XcoUpLeftDownRight);
      XcoSetResizeMethod(colorhole,XcoUpLeftDownRight);
      XcoSetResizeMethod(bcmd,XcoUpRight);
      
      icol=PIXEL(polygonrule[item].ir,polygonrule[item].ig,polygonrule[item].ib);
      set_colorpixmap(icol);

      hole=XcoCreateHole(add_decorbox,5,285,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Ligand atom scaling parameter",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].scalerad);
      scaleraddialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,325,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Central atom scaling parameter",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].centralscalerad);
      centralscaleraddialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,365,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Other atom scaling parameter",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].otherscalerad);
      otherscaleraddialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,405,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Phong rsv",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].rsv);
      rsvdialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,445,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Phong n",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4d",polygonrule[item].n);
      phongndialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      hole=XcoCreateHole(add_decorbox,5,485,ADD_WINDOW_WIDTH-20,40);
      hole2=XcoCreateHole(hole,0,0,ADD_WINDOW_WIDTH-20,50);
      XcoSetResizeMethod(hole2,XcoUpResize);
      label=XcoCreateLabel(hole2,0,0,"Transparency",0,0);
      XcoSetResizeMethod(label,XcoUpLeft);
      ypos=XcoGetObjectHeight(label);
      sprintf(rmindistances,"%4.2f",polygonrule[item].trans);
      transdialog=XcoCreateDialog(hole2,0,ypos,add_window,rmindistances,4,80,0);

      okbutton=XcoCreateCommand(add_window,10,
				ADD_WINDOW_HEIGHT-30,okbuttonname,0,0);
      XcoSetResizeMethod(okbutton,XcoDownLeft);
      XcoAddCallback(okbutton,okbutton_callback);

      xpos=20+XcoGetObjectWidth(okbutton);
      cancelbutton=XcoCreateCommand(add_window,xpos,
					      ADD_WINDOW_HEIGHT-30,
					      "Cancel",0,0);
      XcoSetResizeMethod(cancelbutton,XcoDownLeft);
      XcoAddCallback(cancelbutton,cancelbutton_callback);



      add_rule_open=1;
    }
  else
    {
      XRaiseWindow(XcoGetDisplay(),XcoWindow(add_window));
    }
}

static void add_polygonrule()
{
  if (!add_rule_open)
    {
      delete_if_cancel=1;
      numrules++;
      strcpy(polygonrule[numrules-1].name,"New rule");
      polygonrule[numrules-1].ncentral=0;
      polygonrule[numrules-1].nligand=0;
      polygonrule[numrules-1].r_central_ligand=3.;
      polygonrule[numrules-1].r_ligand_ligand=5.;
      polygonrule[numrules-1].r_min_distance=0.1;
      polygonrule[numrules-1].rsv=0.7;
      polygonrule[numrules-1].n=20;
      polygonrule[numrules-1].scalerad=0.1;
      polygonrule[numrules-1].centralscalerad=0.5;
      polygonrule[numrules-1].otherscalerad=1.0;
      polygonrule[numrules-1].trans=0.7;
      polygonrule[numrules-1].ir=POLY_DEFAULT_R;
      polygonrule[numrules-1].ig=POLY_DEFAULT_G;
      polygonrule[numrules-1].ib=POLY_DEFAULT_B;
      
      modify_polygonrule(numrules-1,"Add polygonrule","Add");
      delete_list();
      make_list();
      XcoSetListData(list,list_items,list_no);
    }
  else
    {
      XRaiseWindow(XcoGetDisplay(),XcoWindow(add_window));
    }
}

static void polygon_add(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      add_polygonrule();
    }
}

static void polygon_modify(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int status=XcoGetListStatus(list);
      if (status!=-1)
	{
	  if (!add_rule_open)
	    {
	      delete_if_cancel=0;
	      modify_polygonrule(status,"Modify polygonrule","Modify");
	    }
	}
    }
}

static void update_em()
{
  if (add_rule_open)
    close_add_rule_window();    
  if (polygon_rules_open)
    {
      delete_list();
      make_list();
      XcoSetListData(list,list_items,list_no);
    }
}

void polygon_rules()
{
  if (!polygon_rules_open)
    {
      XcoObject apply,apply2,add,modify,del,up,down;
      int xpos;
      
      if (virgin)
	{
	  register_update_function(update_em);
	  virgin=0;
	}

      polygon_rules_open=1;
      make_list();
      top_window=XcoCreateNamedWindow(0,0,WINDOW_WIDTH,WINDOW_HEIGHT,DEFAULT_BACKGROUND,1,-1,"Polygon rules");
      XcoAddCallback(top_window,polygon_delete_window);
      decorbox=XcoCreateBox3D(top_window,5,5,WINDOW_WIDTH-10,WINDOW_HEIGHT-75,2,1);
      list=XcoCreateList(decorbox,5,5,WINDOW_WIDTH-20,WINDOW_HEIGHT-90,
			 20,list_items,list_no);
      XcoSetResizeMethod(list,XcoUpLeftDownRight);
      
      apply=XcoCreateCommand(top_window,10,WINDOW_HEIGHT-60,"Apply rules",0,0);
      XcoSetResizeMethod(apply,XcoDownLeft);
      XcoAddCallback(apply,polygon_apply);

      apply2=XcoCreateCommand(top_window,15+XcoGetObjectWidth(apply),WINDOW_HEIGHT-60,"Apply rules to current frame",0,0);
      XcoSetResizeMethod(apply2,XcoDownLeft);
      XcoAddCallback(apply2,polygon_apply_frame);

      add=XcoCreateCommand(top_window,10,WINDOW_HEIGHT-30,"Add",0,0);
      XcoSetResizeMethod(add,XcoDownLeft);
      XcoAddCallback(add,polygon_add);
      
      xpos=10+XcoGetObjectWidth(add)+XcoGetObjectX(add);
      modify=XcoCreateCommand(top_window,xpos,WINDOW_HEIGHT-30,"Modify",0,0);
      XcoSetResizeMethod(modify,XcoDownLeft);
      XcoAddCallback(modify,polygon_modify);

      xpos=10+XcoGetObjectWidth(modify)+XcoGetObjectX(modify);
      del=XcoCreateCommand(top_window,xpos,WINDOW_HEIGHT-30,"Delete",0,0);
      XcoSetResizeMethod(del,XcoDownLeft);
      XcoAddCallback(del,polygon_delete);

      xpos=10+XcoGetObjectWidth(del)+XcoGetObjectX(del);
      up=XcoCreateCommand(top_window,xpos,WINDOW_HEIGHT-30,"Up",0,0);
      XcoSetResizeMethod(up,XcoDownLeft);
      XcoAddCallback(up,polygon_up);

      xpos=10+XcoGetObjectWidth(up)+XcoGetObjectX(up);
      down=XcoCreateCommand(top_window,xpos,WINDOW_HEIGHT-30,"Down",0,0);
      XcoSetResizeMethod(down,XcoDownLeft);
      XcoAddCallback(down,polygon_down);

      polygon_rules_open=1;
    }
  else
    {
      XRaiseWindow(XcoGetDisplay(),XcoWindow(top_window));
    }
}
