/* This is window.c
   A part of the Xco library
   Copyright (C) 1997-1998 Daniel Spangberg
   */

static char rcsid[]="$Id: window.c 15 2012-01-23 17:10:06Z daniels $";


#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "Xco.h"

static int tile=0;
static Pixmap pixmaptile;


void __window_callback(XcoObject id,XEvent event)
{
  switch (event.type)
    {
    case ButtonPress:
      if (__objects[id]->menuwidget==0)
	XcoRemoveAnyMenu(-1,0);
      break;
    case ConfigureNotify:
      {
	int width=event.xconfigure.width;
	int height=event.xconfigure.height;
	XcoIntelliResize(id,width,height);
      }
      break;
    }
}

void XcoWindowTile(Pixmap pixmap)
{
  tile=1;
  pixmaptile=pixmap;
}

void XcoWindowNoTile()
{
  tile=0;
}


XcoObject XcoCreateNamedWindow(int x,int y,int width,int height,
			       int backgroundpixel,
			       int managed,
			       XcoObject parent,
			       char *name)
{
  Bool override;
  Window parentWindow;
  XSetWindowAttributes attributes;
  unsigned long valuemask;
  XSizeHints *sizehints=XAllocSizeHints();
  XWMHints *wmhints=XAllocWMHints();
  XClassHint *classhints=XAllocClassHint();
  XTextProperty textproperty;
  XGCValues xgcvalues;
  XcoObject myobject;
  int screen_height,screen_width;
  screen_height=XcoGetScreenHeight();
  screen_width=XcoGetScreenWidth();

  /* Make sure the window is at least visible... */
  if (width>screen_width)
  {
      /* At least try to center it... */
      x=-(width-screen_width)/2;
  }
  else
  {
      if (x+width>screen_width)
      {
	  /* Move the window... */
	  x=screen_width-width;
      }
  }

  if (height>screen_height)
  {
      /* At least try to center it... */
      y=-(height-screen_height)/2;
  }
  else
  {
      if (y+height>screen_height)
      {
	  /* Move the window... */
	  y=screen_height-height;
      }
  }


  myobject=XcoCreateObject(parent,XcoTWindow,
				     x,y,width,height);

  __objects[myobject]->background=backgroundpixel;
  

  override=False;
  if (!managed)
    override=True;


  if (parent==-1)
    {
      parentWindow=RootWindow(XcoGetDisplay(),DefaultScreen(XcoGetDisplay()));
    }
  else
    {
      parentWindow=__objects[parent]->window;
    }

  /* grey pixel */
  attributes.background_pixel=XcoGetPixel32(0,0,__objects[myobject]->background);
  attributes.border_pixel=XcoGetPixel32(0,0,0);
  attributes.colormap=XcoGetColormap();
  if (parent==-1)
    attributes.event_mask=ButtonPressMask|ExposureMask|StructureNotifyMask;
  else
    attributes.event_mask=ButtonPressMask|ExposureMask;

  attributes.override_redirect=override;
  valuemask=CWBackPixel|CWBorderPixel|CWEventMask|CWOverrideRedirect;
  if (XcoWParamsQuery())
    valuemask|=CWColormap;

  __objects[myobject]->window=XCreateWindow(XcoGetDisplay(),
					    parentWindow,x,y,width,height,0,
					    XcoGetDepth(),
					    InputOutput,XcoGetVisual(),
					    valuemask,&attributes);

  sizehints->flags=USSize|PSize;
  sizehints->x=x;
  sizehints->y=y;
  sizehints->width=width;
  sizehints->height=height;
  XSetNormalHints(XcoGetDisplay(),__objects[myobject]->window,sizehints);
  
  wmhints->flags=InputHint|StateHint;
  wmhints->input=True;
  wmhints->initial_state=NormalState;
  XSetWMHints(XcoGetDisplay(),__objects[myobject]->window,wmhints);

  if ((XcoGetApplicationClass()) && (XcoGetApplicationName()))
    {
      classhints->res_name=XcoGetApplicationName();
      classhints->res_class=XcoGetApplicationClass();
      XSetClassHint(XcoGetDisplay(),__objects[myobject]->window,classhints);
    }

  XStringListToTextProperty(&name,1,&textproperty);
  XSetWMName(XcoGetDisplay(),__objects[myobject]->window,&textproperty);

  XMapWindow(XcoGetDisplay(),__objects[myobject]->window);

  if (parent==-1)
    {
      __objects[myobject]->dw=XInternAtom(XcoGetDisplay(),
					  "WM_DELETE_WINDOW", True);
      XSetWMProtocols(XcoGetDisplay(), __objects[myobject]->window,
		      &__objects[myobject]->dw, 1);

    }
  XcoAddCallback(myobject,__window_callback);
  valuemask=0;

  if (parent==-1)
    while(1) {
      XEvent event;
      XNextEvent(XcoGetDisplay(), &event);
      if (event.type == MapNotify)
	break;
    }

  __objects[myobject]->gc=XCreateGC(XcoGetDisplay(),
				    __objects[myobject]->window,
				    valuemask,&xgcvalues);

  if (tile)
    XcoSetBackgroundPixmap(myobject, pixmaptile,True);

  XFree(classhints);
  XFree(wmhints);
  XFree(sizehints);
  return(myobject);
}

XcoObject XcoCreateWindow(int x,int y,int width,int height,
			  int backgroundpixel,
			  int managed,
			  XcoObject parent)
{
  return XcoCreateNamedWindow(x,y,width,height,
			      backgroundpixel,
			      managed,
			      parent,"");
}

int XcoDeleteWindow(XcoObject id,XEvent event)
{
  int r=0;
  if (event.type==ClientMessage)
    {
      if (__objects[id]->dw==(Atom) event.xclient.data.l[0])
	r=1;
    }

  return r;
}

void XcoSetBackground(XcoObject id, int background)
{
  __objects[id]->background=background;
  XSetWindowBackground(XcoGetDisplay(),__objects[id]->window,XcoGetPixel32(0,0,background));
  XClearArea(XcoGetDisplay(),__objects[id]->window,0,0,0,0,True);
}

void XcoSetBackgroundPixmap(XcoObject id, Pixmap pixmap,Bool exposure)
{
  XSetWindowBackgroundPixmap(XcoGetDisplay(),XcoWindow(id),pixmap);
  XClearArea(XcoGetDisplay(),__objects[id]->window,0,0,0,0,exposure);
}

int XcoGetBackground(XcoObject id)
{
  return (__objects[id]->background);
}

void XcoSetForeground(XcoObject id, int foreground)
{
  __objects[id]->foreground=foreground;
  XClearArea(XcoGetDisplay(),__objects[id]->window,0,0,0,0,True);
}

void XcoMoveWindow(XcoObject id,int x,int y)
{
  __objects[id]->x=x;
  __objects[id]->y=y;
  XMoveWindow(XcoGetDisplay(),__objects[id]->window,x,y);
}

void XcoMoveResizeWindow(XcoObject id,int x,int y,int width,int height)
{
  __objects[id]->x=x;
  __objects[id]->y=y;
  if (width<=0)
    width=1;
  if (height<=0)
    height=1;
  __objects[id]->width=width;
  __objects[id]->height=height;
  XMoveResizeWindow(XcoGetDisplay(),__objects[id]->window,x,y,width,height);
}

void XcoIntelliResize(XcoObject id,int width,int height)
{
  int i,x,y,new_width,new_height,xmove,ymove;
  for (i=0; i<MAX_OBJECTS; i++)
    {
      if (__used_objects[i])
	{
	  if (__objects[i]->parent==id)
	    {
	      switch(__objects[i]->resize)
		{
		case XcoResize:
		  {
		    x=(__objects[i]->orig_x*width)/__objects[id]->orig_width;
		    y=(__objects[i]->orig_y*height)/__objects[id]->orig_height;
		    new_width=(__objects[i]->orig_width*width)/__objects[id]->orig_width;
		    new_height=(__objects[i]->orig_height*height)/__objects[id]->orig_height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		case XcoUpLeft:
		  /* Nothing! */
		  break;
		case XcoUpRight:
		  {
		    xmove=width-__objects[id]->orig_width;
		    ymove=0;
		    x=__objects[i]->orig_x+xmove;
		    y=__objects[i]->orig_y+ymove;
		    XcoMoveWindow(i,x,y);
		  }
		  break;
		case XcoDownRight:
		  {
		    xmove=width-__objects[id]->orig_width;
		    ymove=height-__objects[id]->orig_height;
		    x=__objects[i]->orig_x+xmove;
		    y=__objects[i]->orig_y+ymove;
		    XcoMoveWindow(i,x,y);
		  }
		  break;
		case XcoDownLeft:
		  {
		    xmove=0;
		    ymove=height-__objects[id]->orig_height;
		    x=__objects[i]->orig_x+xmove;
		    y=__objects[i]->orig_y+ymove;
		    XcoMoveWindow(i,x,y);
		  }
		  break;
		case XcoUpResize:
		  {
		    x=(__objects[i]->orig_x*width)/__objects[id]->orig_width;
		    ymove=0;
		    y=__objects[i]->orig_y+ymove;
		    new_width=(__objects[i]->orig_width*width)/__objects[id]->orig_width;
		    new_height=__objects[i]->height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		case XcoDownResize:
		  {
		    x=(__objects[i]->orig_x*width)/__objects[id]->orig_width;
		    ymove=height-__objects[id]->orig_height;
		    y=__objects[i]->orig_y+ymove;
		    new_width=(__objects[i]->orig_width*width)/__objects[id]->orig_width;
		    new_height=__objects[i]->height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		case XcoResizeLeft:
		  {
		    xmove=0;
		    x=__objects[i]->orig_x+xmove;
		    y=(__objects[i]->orig_y*height)/__objects[id]->orig_height;
		    new_width=__objects[i]->width;
		    new_height=(__objects[i]->orig_height*height)/__objects[id]->orig_height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		case XcoResizeRight:
		  {
		    xmove=width-__objects[id]->orig_width;
		    x=__objects[i]->orig_x+xmove;
		    y=(__objects[i]->orig_y*height)/__objects[id]->orig_height;
		    new_width=__objects[i]->width;
		    new_height=(__objects[i]->orig_height*height)/__objects[id]->orig_height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		case XcoUpLeftDownRight:
		  {
		    x=__objects[i]->orig_x;
		    y=__objects[i]->orig_y;
		    new_width=__objects[i]->orig_width+width-__objects[id]->orig_width;
		    new_height=__objects[i]->orig_height+height-__objects[id]->orig_height;
		    XcoIntelliResize(i,new_width,new_height);
		    XcoMoveResizeWindow(i,x,y,new_width,new_height);
		  }
		  break;
		}
	    }
	}
    }
  __objects[id]->width=width;
  __objects[id]->height=height;
}

